package weather

import (
	"context"
	"fmt"
	"time"

	"a.yandex-team.ru/travel/library/go/renderer"
	"a.yandex-team.ru/travel/library/go/tanker"
	"a.yandex-team.ru/travel/notifier/internal/constants"
	"a.yandex-team.ru/travel/notifier/internal/models"
	"a.yandex-team.ru/travel/notifier/internal/orders"
	"a.yandex-team.ru/travel/notifier/internal/service/pretrip/blocks"
	"a.yandex-team.ru/travel/notifier/internal/service/pretrip/blocks/ui"
	"a.yandex-team.ru/travel/notifier/internal/service/pretrip/interfaces"
	"a.yandex-team.ru/travel/notifier/internal/structs"
)

const (
	weatherIconsPatternURL   = "https://yastatic.net/weather/i/icons/funky/png/dark/48/%s.png"
	numberOfDaysWithForecast = 6
	isoDateFormat            = "2006-01-02"
)

var monthNames = []string{"", "янв", "фев", "мар", "апр", "май", "июн", "июл", "авг", "сен", "окт", "ноя", "дек"}

type Provider struct {
	texts                  tanker.Keyset
	routePointsExtractor   interfaces.RoutePointsExtractor
	weatherProvider        interfaces.WeatherProvider
	settlementDataProvider interfaces.SettlementDataProvider
}

func NewProvider(
	texts tanker.Keyset,
	routePointsExtractor interfaces.RoutePointsExtractor,
	weatherProvider interfaces.WeatherProvider,
	cityDataProvider interfaces.SettlementDataProvider,
) *Provider {
	provider := &Provider{
		texts:                  texts,
		routePointsExtractor:   routePointsExtractor,
		weatherProvider:        weatherProvider,
		settlementDataProvider: cityDataProvider,
	}
	return provider
}

func (p *Provider) GetBlock(ctx context.Context, orderInfo *orders.OrderInfo, notification models.Notification) (renderer.Block, error) {
	block := ui.NewWeatherBlock()
	order, err := orderInfo.ToOrder()
	if err != nil {
		return block, err
	}
	settlementID, err := p.routePointsExtractor.ExtractArrivalSettlementID(orderInfo)
	if err != nil {
		return block, err
	}
	if settlementID <= 0 {
		return block, blocks.ErrUnknownSettlement
	}

	var title string
	switch notification.Subtype {
	case models.NotificationWeekBefore:
		settlementTitleTranslation, found := p.settlementDataProvider.GetTitleTranslation(settlementID)
		if !found {
			return block, blocks.ErrNoSettlementTitleTranslation
		}
		textParams := make(map[string]interface{})
		textParams["settlementTitle"] = settlementTitleTranslation.Prepositional
		textParams["settlementPreposition"] = settlementTitleTranslation.LocativePreposition
		title, err = tanker.TemplateToString("week.title", p.texts.GetSingular("week.title", "ru"), textParams)
		if err != nil {
			return block, err
		}
	case models.NotificationDayBefore:
		fallthrough
	case models.NotificationAdhoc:
		title = p.texts.GetSingular("adhoc.title", "ru")
	default:
		return block, fmt.Errorf("unknown notification subtype")
	}

	if title == "" {
		return block, fmt.Errorf("empty block title")
	}
	block.Title = title

	geoID, found := p.settlementDataProvider.GetGeoID(settlementID)
	if !found || geoID == 0 {
		return block, fmt.Errorf("unable to get settlement geoID")
	}

	weather, err := p.weatherProvider.GetWeather(ctx, geoID, "ru")
	if err != nil {
		return block, fmt.Errorf("unable to get weather for city: %w", err)
	}

	if weather.Info == nil {
		return block, fmt.Errorf("unable to verify the city ID for the weather")
	}

	settlementGeoID, found := p.settlementDataProvider.GetGeoID(settlementID)
	if found && weather.Info.GeoID != settlementGeoID {
		return block, fmt.Errorf("geoID %d and weather geoID %d don't match", settlementID, weather.Info.GeoID)
	}

	block.Action = ui.SecondaryAction{
		Text:  p.texts.GetSingular("detailed.forecast", "ru"),
		Theme: ui.SecondaryActionTheme,
		URL:   weather.Info.URL,
	}

	if order.ArrivalDate.IsZero() {
		return block, fmt.Errorf("unable to get arrival date for the order")
	}
	tz := p.settlementDataProvider.GetSettlementTimeZone(settlementID)
	if tz == nil {
		return block, fmt.Errorf("unable to get destination timezone for the order")
	}
	forecasts, err := p.getBlockItems(weather.Forecasts, order.ArrivalDate.In(tz))
	block.Items = forecasts
	return block, err
}

func (p *Provider) getBlockItems(forecasts []structs.WeatherForecast, startDate time.Time) ([]ui.WeatherBlockItem, error) {
	result := make([]ui.WeatherBlockItem, 0, numberOfDaysWithForecast)
	// find the first date of the forecast
	strStartDate := startDate.Format(isoDateFormat)
	startIndex := -1
	for index, f := range forecasts {
		if f.Date == strStartDate {
			startIndex = index
		}
	}

	if startIndex == -1 {
		return result, fmt.Errorf("unable to obtain a forecast for the start date %v", startDate)
	}

	for index, f := range forecasts {
		// we only have up to 10 elems, cpu loss is negligible and the code is simpler
		if index < startIndex || index >= index+numberOfDaysWithForecast {
			continue
		}
		blockItem, err := p.getBlockItem(f, len(result) == 0)
		if err != nil {
			return result, fmt.Errorf("unable to render forecast for date %s, error: %w", f.Date, err)
		}
		result = append(result, blockItem)
	}
	return result, nil
}

func (p *Provider) getBlockItem(forecast structs.WeatherForecast, isFirstItem bool) (ui.WeatherBlockItem, error) {
	item := ui.WeatherBlockItem{}
	forecastDate, err := time.Parse(isoDateFormat, forecast.Date)
	if err != nil {
		return item, err
	}
	item.Date = fmt.Sprintf("%d %s", forecastDate.Day(), monthNames[int(forecastDate.Month())])
	item.IsWeekend = forecastDate.Weekday() == time.Saturday || forecastDate.Weekday() == time.Sunday
	item.Day = constants.WeekdayShortNames[int(forecastDate.Weekday())]

	if forecast.DayParts == nil {
		return item, fmt.Errorf("no day parts in the forecast")
	}

	dayPart, hasDayPart := forecast.DayParts["day"]
	if !hasDayPart {
		return item, fmt.Errorf("no day temperature in the forecast")
	}

	nightPart, hasNightPart := forecast.DayParts["night"]
	if !hasNightPart {
		return item, fmt.Errorf("no night temperature in the forecast")
	}

	if isFirstItem {
		item.Temperature.Day, err = p.getTemperatureString(dayPart.TempAvg, "day.temp")
		if err != nil {
			return item, fmt.Errorf("invalid day temperature template")
		}
		item.Temperature.Night, err = p.getTemperatureString(nightPart.TempAvg, "night.temp")
		if err != nil {
			return item, fmt.Errorf("invalid night temperature template")
		}
	} else {
		item.Temperature.Day = getTemperatureValue(dayPart.TempAvg)
		item.Temperature.Night = getTemperatureValue(nightPart.TempAvg)
	}

	if dayPart.Icon == "" {
		return item, fmt.Errorf("no day weather icon in the forecast")
	}
	if dayPart.Condition == "" {
		return item, fmt.Errorf("no day weather condition in the forecast")
	}

	item.Conditions.Icon = fmt.Sprintf(weatherIconsPatternURL, dayPart.Icon)
	item.Conditions.Description = p.texts.GetSingular(dayPart.Condition, "ru")
	if item.Conditions.Description == "" {
		return item, fmt.Errorf("unexpected weather condition in the forecast: %s", dayPart.Condition)
	}
	return item, nil
}

func (p *Provider) getTemperatureString(temperature int, templateName string) (string, error) {
	textParams := make(map[string]interface{})
	textParams["degrees"] = getTemperatureValue(temperature)
	return tanker.TemplateToString(templateName, p.texts.GetSingular(templateName, "ru"), textParams)
}

func getTemperatureValue(temperature int) string {
	degrees := "0"
	if temperature != 0 {
		degrees = fmt.Sprintf("%+d", temperature)
	}
	return degrees
}

func (p *Provider) GetBlockType() blocks.BlockType {
	return blocks.WeatherBlock
}
