package pretrip

import (
	"time"

	"a.yandex-team.ru/travel/notifier/internal/externalhttp"
	"a.yandex-team.ru/travel/notifier/internal/models"
	"a.yandex-team.ru/travel/notifier/internal/rollout"
	"a.yandex-team.ru/travel/notifier/internal/service/pretrip/blocks"
	"a.yandex-team.ru/travel/notifier/internal/service/pretrip/blocks/audioguide"
	"a.yandex-team.ru/travel/notifier/internal/service/pretrip/blocks/beforedeparture"
	"a.yandex-team.ru/travel/notifier/internal/service/pretrip/blocks/disclaimers"
	"a.yandex-team.ru/travel/notifier/internal/service/pretrip/blocks/disk"
	"a.yandex-team.ru/travel/notifier/internal/service/pretrip/blocks/events"
	"a.yandex-team.ru/travel/notifier/internal/service/pretrip/blocks/footer"
	"a.yandex-team.ru/travel/notifier/internal/service/pretrip/blocks/greeting"
	"a.yandex-team.ru/travel/notifier/internal/service/pretrip/blocks/header"
	"a.yandex-team.ru/travel/notifier/internal/service/pretrip/blocks/hotelorder"
	"a.yandex-team.ru/travel/notifier/internal/service/pretrip/blocks/hotels"
	"a.yandex-team.ru/travel/notifier/internal/service/pretrip/blocks/movies"
	"a.yandex-team.ru/travel/notifier/internal/service/pretrip/blocks/navigation"
	"a.yandex-team.ru/travel/notifier/internal/service/pretrip/blocks/rides"
	"a.yandex-team.ru/travel/notifier/internal/service/pretrip/blocks/testing"
	"a.yandex-team.ru/travel/notifier/internal/service/pretrip/blocks/trainorder"
	"a.yandex-team.ru/travel/notifier/internal/service/pretrip/blocks/weather"
	"a.yandex-team.ru/travel/notifier/internal/service/pretrip/interfaces"
)

type NotificationConfig struct {
	Blocks []blocks.BlockConfig
}

type TestingConfig struct {
	Emails                    []string      `config:"PRETRIP_TESTING_EMAILS" yaml:"emails"`
	AdhocSendingInterval      time.Duration `config:"PRETRIP_TESTING_ADHOC_SENDING_INTERVAL" yaml:"adhoc_sending_interval"`
	WeekBeforeSendingInterval time.Duration `config:"PRETRIP_TESTING_WEEK_BEFORE_SENDING_INTERVAL" yaml:"week_before_sending_interval"`
	DayBeforeSendingInterval  time.Duration `config:"PRETRIP_TESTING_DAY_BEFORE_SENDING_INTERVAL" yaml:"day_before_sending_interval"`
}

type BlockProvidersConfig struct {
	AudioGuidesProvider                externalhttp.Config `yaml:"audioguides"`
	AudioGuidesProviderTimeout         time.Duration       `config:"PRETRIP_AUDIOGUIDE_TIMEOUT" yaml:"audioguides_timeout"`
	EventsProvider                     externalhttp.Config `yaml:"events"`
	EventsProviderTimeout              time.Duration       `config:"PRETRIP_EVENTS_TIMEOUT" yaml:"events_timeout"`
	WeatherProvider                    externalhttp.Config `yaml:"weather"`
	WeatherProviderTimeout             time.Duration       `config:"PRETRIP_WEATHER_TIMEOUT" yaml:"weather_timeout"`
	HotelsProvider                     externalhttp.Config `yaml:"hotels"`
	HotelsProviderTimeout              time.Duration       `config:"PRETRIP_HOTELS_TIMEOUT" yaml:"hotels_timeout"`
	AdditionalOrderInfo                externalhttp.Config `yaml:"additional_order_info"`
	AdditionalOrderInfoProviderTimeout time.Duration       `config:"PRETRIP_ADDITIONAL_ORDER_INFO_TIMEOUT" yaml:"additional_order_info_timeout"`
}

type PlanningConfig struct {
	Testing              TestingConfig
	SettlementsWhitelist []int              `yaml:"settlements_whitelist"`
	EnabledOrderTypes    []string           `config:"PRETRIP_ENABLED_ORDER_TYPES" yaml:"enabled_order_types"`
	EmailRules           rollout.EmailRules `yaml:"email_rules"`
	TrainMinDistanceKM   float64            `config:"PRETRIP_TRAIN_MIN_DISTANCE_KM" yaml:"train_min_distance_km"`
}

type ProcessingConfig struct {
	SenderCampaigns            map[models.NotificationSubtype]string   `yaml:"sender_campaigns"`
	PostponeInterval           time.Duration                           `yaml:"postpone_interval"`
	NotificationConfigByType   map[models.OrderType]NotificationConfig `yaml:"notification_config_by_type"`
	DBStatementTimeout         time.Duration                           `config:"PRETRIP_DB_STATEMENT_TIMEOUT" yaml:"db_statement_timeout"`
	DBLockTimeout              time.Duration                           `config:"PRETRIP_DB_LOCK_TIMEOUT" yaml:"db_lock_timeout"`
	DBIdleInTransactionTimeout time.Duration                           `config:"PRETRIP_DB_IDLE_IN_TRANSACTION_TIMEOUT" yaml:"db_idle_in_transaction_timeout"`
	WaitForLockBeforeSending   bool                                    `config:"PRETRIP_WAIT_FOR_LOCK_BEFORE_SENDING" yaml:"wait_for_lock_before_sending"`
	DeduplicateByRecipient     bool                                    `config:"PRETRIP_DEDUPLICATE_BY_RECIPIENT" yaml:"deduplicate_by_recipient"`
	BlockProviders             BlockProvidersConfig                    `yaml:"block_providers"`
	SettlementsWhitelist       []int                                   `yaml:"settlements_whitelist"`
	Testing                    TestingConfig
	EnabledOrderTypes          []string           `config:"PRETRIP_ENABLED_ORDER_TYPES" yaml:"enabled_order_types"`
	EmailRules                 rollout.EmailRules `yaml:"email_rules"`
	TrainMinDistanceKM         float64            `config:"PRETRIP_TRAIN_MIN_DISTANCE_KM" yaml:"train_min_distance_km"`
}

var defaultBlockProvidersConfig = BlockProvidersConfig{
	AudioGuidesProviderTimeout:         6 * time.Second,
	EventsProviderTimeout:              6 * time.Second,
	HotelsProviderTimeout:              6 * time.Second,
	WeatherProviderTimeout:             6 * time.Second,
	AdditionalOrderInfoProviderTimeout: 6 * time.Second,
}

var (
	DefaultPlanningConfig = PlanningConfig{
		SettlementsWhitelist: nil,
		Testing:              TestingConfig{Emails: make([]string, 0)},
		EnabledOrderTypes:    []string{models.OrderHotel.String(), models.OrderTrain.String()},
		EmailRules:           rollout.EmailRules{},
		TrainMinDistanceKM:   200,
	}

	DefaultProcessingConfig = ProcessingConfig{
		SenderCampaigns: map[models.NotificationSubtype]string{
			models.NotificationDayBefore:  "08MG1NW3-UUQ",
			models.NotificationWeekBefore: "08MG1NW3-UUQ",
			models.NotificationAdhoc:      "08MG1NW3-UUQ",
		},
		PostponeInterval:           10 * time.Minute,
		NotificationConfigByType:   make(map[models.OrderType]NotificationConfig),
		DBStatementTimeout:         60 * time.Second,
		DBLockTimeout:              10 * time.Second,
		DBIdleInTransactionTimeout: 60 * time.Second,
		WaitForLockBeforeSending:   true,
		Testing:                    TestingConfig{Emails: make([]string, 0)},
		BlockProviders:             defaultBlockProvidersConfig,
		SettlementsWhitelist:       make([]int, 0),
		DeduplicateByRecipient:     true,
		EnabledOrderTypes:          []string{models.OrderHotel.String(), models.OrderTrain.String()},
		TrainMinDistanceKM:         200,
	}
)

// TODO: Add all blocks for Pretrip-notification https://st.yandex-team.ru/RASPTICKETS-19808
func GetDefaultBlockProviders(
	travelPortalURL string,
	cityImagesProvider interfaces.CityImagesProvider,
	audioGuidesProvider interfaces.AudioGuidesProvider,
	eventsProvider interfaces.EventsProvider,
	weatherProvider interfaces.WeatherProvider,
	routePointsExtractor interfaces.RoutePointsExtractor,
	unsubscribeLinkGenerator interfaces.UnsubscribeLinkGenerator,
	settlementDataProvider interfaces.SettlementDataProvider,
	stationDataProvider interfaces.StationDataProvider,
	timeZoneDataProvider interfaces.TimeZoneDataProvider,
	hotelsProvider interfaces.HotelsProvider,
	additionalOrderInfoClient interfaces.AdditionalOrderInfoProvider,
) []blocks.CollectorOption {
	return []blocks.CollectorOption{
		blocks.WithBlockProvider(header.NewProvider()),
		blocks.WithBlockProvider(
			greeting.NewProvider(
				greeting.DefaultKeyset,
				routePointsExtractor,
				cityImagesProvider,
				settlementDataProvider,
			),
		),
		blocks.WithBlockProvider(rides.NewProvider(rides.DefaultKeyset, routePointsExtractor)),
		blocks.WithBlockProvider(navigation.NewProvider(navigation.DefaultKeyset, routePointsExtractor)),
		blocks.WithBlockProvider(testing.NewProvider(routePointsExtractor, settlementDataProvider)),
		blocks.WithBlockProvider(footer.NewProvider(footer.DefaultKeyset, unsubscribeLinkGenerator)),
		blocks.WithBlockProvider(
			audioguide.NewProvider(audioguide.DefaultKeyset, routePointsExtractor, audioGuidesProvider, settlementDataProvider),
		),
		blocks.WithBlockProvider(movies.NewProvider(movies.DefaultKeyset)),
		blocks.WithBlockProvider(disk.NewProvider(disk.DefaultKeyset)),
		blocks.WithBlockProvider(beforedeparture.NewProvider(beforedeparture.DefaultKeyset)),
		blocks.WithBlockProvider(
			events.NewProvider(events.DefaultKeyset, routePointsExtractor, eventsProvider, settlementDataProvider),
		),
		blocks.WithBlockProvider(
			weather.NewProvider(
				weather.DefaultKeyset,
				routePointsExtractor,
				weatherProvider,
				settlementDataProvider,
			),
		),
		blocks.WithBlockProvider(
			hotels.NewProvider(
				hotels.DefaultKeyset,
				routePointsExtractor,
				hotelsProvider,
				settlementDataProvider,
				travelPortalURL,
			),
		),
		blocks.WithBlockProvider(hotelorder.NewProvider(hotelorder.DefaultKeyset, additionalOrderInfoClient)),
		blocks.WithBlockProvider(
			trainorder.NewProvider(
				trainorder.DefaultKeyset,
				stationDataProvider,
				timeZoneDataProvider,
				additionalOrderInfoClient,
				travelPortalURL,
			),
		),
		blocks.WithDisclaimersBlockProvider(disclaimers.NewProvider(disclaimers.DefaultKeyset)),
	}
}
