package interfaces

import (
	"context"
	"time"

	"a.yandex-team.ru/travel/library/go/renderer"
	"a.yandex-team.ru/travel/library/go/sender"
	contentadmin "a.yandex-team.ru/travel/marketing/content/v1"
	"a.yandex-team.ru/travel/notifier/internal/models"
	"a.yandex-team.ru/travel/notifier/internal/orders"
	"a.yandex-team.ru/travel/notifier/internal/service/pretrip/blocks"
	"a.yandex-team.ru/travel/notifier/internal/service/pretrip/logging/renderlog/record"
	"a.yandex-team.ru/travel/notifier/internal/structs"
	"a.yandex-team.ru/travel/notifier/internal/travelapi"
	basicinfo "a.yandex-team.ru/travel/orders/proto/services/orders/basic_info/v1"
	travel_commons_proto "a.yandex-team.ru/travel/proto"
	"a.yandex-team.ru/travel/proto/dicts/rasp"
)

type Renderer interface {
	RenderStructured(context.Context, []renderer.Block) (*renderer.StructuredHTML, error)
}

type Sender interface {
	SendTransactional(ctx context.Context, sendRequest sender.TransactionalRequest) ([]byte, error)
}

type NotificationDeadlineHandler interface {
	OnNotificationDeadline(
		ctx context.Context,
		notification *models.Notification,
		orderInfo *orders.OrderInfo,
	) error
}

type OrderProvider interface {
	GetOrderInfoByID(context.Context, string) (*orders.OrderInfo, error)
}

type OrderInfoMapper interface {
	Map(*basicinfo.BasicOrderInfo) (*orders.OrderInfo, error)
}

type NotificationsScheduler interface {
	Schedule(context.Context, models.Notification, time.Time) (*models.Notification, error)
	CancelPlannedNotificationsByOrderID(context.Context, string, models.NotificationType, models.DispatchType) ([]models.Notification, error)
	ScheduleMany(context.Context, []models.Notification, time.Time) ([]models.Notification, error)
	AlreadySentForOrder(context.Context, string) ([]models.Notification, error)
}

type RollOutService interface {
	IsEnabledForEmail(string) bool
}

type EmailEnabler interface {
	Allows(string) bool
}

type CityImagesProvider interface {
	GetCityImageByID(context.Context, int) (*contentadmin.CityImage, error)
}

type AudioGuidesProvider interface {
	GetAudioGuides(context.Context, int) (structs.AudioGuidesForCity, error)
}

type EventsProvider interface {
	GetEvents(ctx context.Context, geoID int, startDate time.Time, latitude, longitude float64) (structs.EventsForCity, error)
}

type WeatherProvider interface {
	GetWeather(ctx context.Context, geoID int, lang string) (structs.WeatherForCity, error)
}

type HotelsProvider interface {
	GetHotels(ctx context.Context, geoID int, orderID string, limit int) (*structs.HotelPayload, error)
}

type RoutePointsExtractor interface {
	ExtractDepartureSettlementID(order *orders.OrderInfo) (id int, err error)
	ExtractArrivalSettlementID(order *orders.OrderInfo) (id int, err error)

	ExtractDepartureStationID(order *orders.OrderInfo) (id int, err error)
	ExtractArrivalStationID(order *orders.OrderInfo) (id int, err error)
}

type StationIDExtractor interface {
	ExtractDepartureStation(order *orders.OrderInfo) (id int, err error)
	ExtractArrivalStation(order *orders.OrderInfo) (id int, err error)
}

type UnsubscribeLinkGenerator interface {
	Generate(recipient models.Recipient) string
}

type SettlementTitleTranslationProvider interface {
	GetTitleTranslation(int) (*travel_commons_proto.TTranslationCaseRu, bool)
}

type SettlementGeoIDProvider interface {
	GetGeoID(int) (int, bool)
}

type SettlementNominativeTitleExtractor interface {
	GetNominativeTitle(id int) (title string, found bool)
}

type SettlementAccusativeTitleExtractor interface {
	GetPreposition(id int) string
	GetAccusativeTitleWithPreposition(id int) (title string, found bool)
}

type SettlementTimeZoneProvider interface {
	GetSettlementTimeZone(int) *time.Location
}

type DistanceCalculator interface {
	GetDistance(fromID, toID int) (float64, bool)
}

type SettlementDataProvider interface {
	GetCoordinates(int) (latitude, longitude float64, found bool)
	SettlementTitleTranslationProvider
	SettlementNominativeTitleExtractor
	SettlementAccusativeTitleExtractor
	SettlementGeoIDProvider
	SettlementTimeZoneProvider
	DistanceCalculator
}

type RenderLogger interface {
	Log(message *record.RenderLogRecord) error
}

type AdditionalOrderInfoProvider interface {
	GetAdditionalOrderInfo(ctx context.Context, orderID string) (*travelapi.AdditionalOrderInfo, error)
}

type TimeZoneDataProvider interface {
	Get(timeZoneID int) (*time.Location, bool)
}

type StationDataProvider interface {
	GetStationByID(int) (*rasp.TStation, bool)
	GetSettlementByStationID(int) (*rasp.TSettlement, bool)
	DistanceCalculator
}

type BlocksProvider interface {
	GetBlocks(context.Context, []blocks.BlockConfig, *orders.OrderInfo, models.Notification) ([]renderer.Block, error)
}

type RecipientsRepository interface {
	GetOrCreate(context.Context, models.Recipient) (*models.Recipient, error)
}

type OrdersRepository interface {
	GetByCorrelationID(ctx context.Context, correlationID string) ([]models.Order, error)
	GetByID(ctx context.Context, orderID string) (models.Order, error)
}
