package pretrip

import (
	"context"
	"fmt"

	"github.com/jonboulle/clockwork"

	"a.yandex-team.ru/travel/library/go/metrics"
	"a.yandex-team.ru/travel/notifier/internal/models"
	"a.yandex-team.ru/travel/notifier/internal/orders"
	"a.yandex-team.ru/travel/notifier/internal/service/pretrip/interfaces"
	"a.yandex-team.ru/travel/notifier/internal/service/pretrip/scheduling"
)

type notificationDeadlineHandler struct {
	notificationBuilder    scheduling.NotificationsBuilder
	notificationsScheduler interfaces.NotificationsScheduler
	clock                  clockwork.Clock
}

func NewNotificationDeadlineHandler(
	notificationBuilder scheduling.NotificationsBuilder,
	notificationsScheduler interfaces.NotificationsScheduler,
	clock clockwork.Clock,
) interfaces.NotificationDeadlineHandler {
	return &notificationDeadlineHandler{
		notificationsScheduler: notificationsScheduler,
		notificationBuilder:    notificationBuilder,
		clock:                  clock,
	}
}

// https://st.yandex-team.ru/RASPTICKETS-20094#602f8ad83ecb4c692d7258ec
func (h *notificationDeadlineHandler) OnNotificationDeadline(
	ctx context.Context,
	notification *models.Notification,
	orderInfo *orders.OrderInfo,
) error {
	errorCode := ""
	defer func() {
		if errorCode != "" {
			metrics.GlobalAppMetrics().GetOrCreateCounter(
				pretripMetricsPrefix,
				map[string]string{"code": errorCode},
				notPlannedAfterDeadlineMetricName,
			).Inc()
		}
	}()
	if notification.Subtype == models.NotificationDayBefore || notification.Subtype == models.NotificationAdhoc {
		return nil
	}
	_, err := h.notificationsScheduler.CancelPlannedNotificationsByOrderID(ctx, orderInfo.ID, notification.Type, notification.DispatchType)
	if err != nil {
		errorCode = "cancel_planned_error"
		return fmt.Errorf("couldn't cancel notifications for order %s: %w", orderInfo.ID, err)
	}

	alreadySentNotifications, err := h.notificationsScheduler.AlreadySentForOrder(ctx, orderInfo.ID)
	if err != nil {
		errorCode = "already_sent_error"
		return fmt.Errorf("couldn't verify whether notifications have been sent already for order %s: %s", orderInfo.ID, err)
	}
	if isAlreadySent(alreadySentNotifications) {
		return nil
	}

	order, err := orderInfo.ToOrder()
	if err != nil {
		errorCode = "order_map_error"
		return fmt.Errorf("couldn't parse order info: %w", err)
	}

	notifications, err := h.notificationBuilder.Build(order, *notification.Recipient, h.clock.Now())
	if err != nil {
		errorCode = "build_error"
		return fmt.Errorf("couldn't build notification: %w", err)
	}
	if len(notifications) > 0 {
		if _, err = h.notificationsScheduler.ScheduleMany(ctx, notifications, h.clock.Now()); err != nil {
			errorCode = "schedule_error"
			return fmt.Errorf("couldn't schedule notification: %w", err)
		}
		for _, n := range notifications {
			metrics.GlobalAppMetrics().GetOrCreateCounter(
				pretripMetricsPrefix,
				map[string]string{"subtype": n.Subtype.String()},
				plannedAfterDeadlineNotificationMetricName,
			).Inc()
		}
	}
	return nil
}
