package pretrip

import (
	"context"
	"encoding/json"
	"fmt"
	"strconv"
	"sync"
	"time"

	"github.com/gofrs/uuid"
	"github.com/jonboulle/clockwork"
	"golang.org/x/sync/errgroup"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/ctxlog"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/library/go/slices"
	"a.yandex-team.ru/travel/library/go/containers"
	"a.yandex-team.ru/travel/library/go/metrics"
	"a.yandex-team.ru/travel/library/go/renderer"
	"a.yandex-team.ru/travel/library/go/sender"
	"a.yandex-team.ru/travel/library/go/tanker"
	"a.yandex-team.ru/travel/notifier/internal/constants"
	"a.yandex-team.ru/travel/notifier/internal/database"
	"a.yandex-team.ru/travel/notifier/internal/models"
	"a.yandex-team.ru/travel/notifier/internal/orders"
	"a.yandex-team.ru/travel/notifier/internal/service/pretrip/interfaces"
	"a.yandex-team.ru/travel/notifier/internal/service/pretrip/logging/renderlog/record"
	"a.yandex-team.ru/travel/notifier/internal/service/processing"
)

var (
	ErrNotificationWithNoOrder = xerrors.New("pre-trip notification has no order")
)

type Processor struct {
	logger                             log.Logger
	renderer                           interfaces.Renderer
	senderClient                       interfaces.Sender
	notificationsRepository            processing.NotificationsRepository
	recipientsRepository               interfaces.RecipientsRepository
	ordersRepository                   interfaces.OrdersRepository
	config                             ProcessingConfig
	orderProvider                      interfaces.OrderProvider
	orderInfoMapper                    interfaces.OrderInfoMapper
	blocksProvider                     interfaces.BlocksProvider
	clock                              clockwork.Clock
	txOptions                          database.TransactionOptions
	settlementIDsWhitelist             containers.Set[int]
	routePointsExtractor               interfaces.RoutePointsExtractor
	settlementAccusativeTitleExtractor interfaces.SettlementAccusativeTitleExtractor
	expiredNotificationDeadlineHandler interfaces.NotificationDeadlineHandler
	rollOutService                     interfaces.RollOutService
	renderLogger                       interfaces.RenderLogger
}

type void struct{}

type processingContext struct {
	context.Context
	notification *models.Notification
	tx           database.NotificationTransaction
	orderInfo    *orders.OrderInfo
	order        models.Order
	iterationID  uuid.UUID
}

func NewProcessor(
	logger log.Logger,
	notificationsRepository processing.NotificationsRepository,
	recipientsRepository interfaces.RecipientsRepository,
	ordersRepository interfaces.OrdersRepository,
	renderer interfaces.Renderer,
	senderClient interfaces.Sender,
	orderProvider interfaces.OrderProvider,
	blocksProvider interfaces.BlocksProvider,
	config ProcessingConfig,
	clock clockwork.Clock,
	routePointsExtractor interfaces.RoutePointsExtractor,
	expiredNotificationDeadlineHandler interfaces.NotificationDeadlineHandler,
	rollOutService interfaces.RollOutService,
	renderLogger interfaces.RenderLogger,
	settlementDestinationTitleExtractor interfaces.SettlementAccusativeTitleExtractor,
) *Processor {
	return &Processor{
		logger:                  logger.WithName("PretripProcessor"),
		orderProvider:           orderProvider,
		blocksProvider:          blocksProvider,
		renderer:                renderer,
		senderClient:            senderClient,
		notificationsRepository: notificationsRepository,
		recipientsRepository:    recipientsRepository,
		ordersRepository:        ordersRepository,
		config:                  config,
		clock:                   clock,
		txOptions: database.TransactionOptions{
			StatementTimeout:                config.DBStatementTimeout,
			LockTimeout:                     config.DBLockTimeout,
			IdleInTransactionSessionTimeout: config.DBIdleInTransactionTimeout,
			WaitForLock:                     config.WaitForLockBeforeSending,
		},
		settlementIDsWhitelist:             containers.SetOf(config.SettlementsWhitelist...),
		routePointsExtractor:               routePointsExtractor,
		expiredNotificationDeadlineHandler: expiredNotificationDeadlineHandler,
		rollOutService:                     rollOutService,
		renderLogger:                       renderLogger,
		settlementAccusativeTitleExtractor: settlementDestinationTitleExtractor,
	}
}

func (p *Processor) Process(
	ctx context.Context,
	notification *models.Notification,
	processingTx database.NotificationTransaction,
	iterationID uuid.UUID,
) {
	processingCtx, successful := p.prepareProcessingContext(ctx, notification, processingTx, iterationID)
	if !successful {
		return
	}
	blocks, successful := p.getBlocks(processingCtx)
	if !successful {
		return
	}
	renderedMessage, successful := p.render(processingCtx, blocks)
	if !successful {
		return
	}
	if p.updateNotificationStatus(processingCtx, models.NotificationStatusReadyToSend) != nil {
		return
	}
	senderRequest := p.buildSenderRequest(renderedMessage, notification.Recipient.GetEmail(), processingCtx)

	if processingCtx.tx, successful = p.beginSendingTx(processingCtx); !successful {
		return
	}

	p.send(processingCtx, senderRequest)
	if err := processingCtx.tx.Commit(p.logger); err != nil {
		p.logger.Error(
			"failed to commit transaction after sending notification",
			append(ctxlog.ContextFields(processingCtx), log.Error(err))...,
		)
	}
}

func (p *Processor) getBlocks(processingCtx processingContext) (blocks []renderer.Block, successful bool) {
	blocks, err := p.blocksProvider.GetBlocks(
		processingCtx,
		p.config.NotificationConfigByType[processingCtx.order.Type].Blocks,
		processingCtx.orderInfo,
		*processingCtx.notification,
	)
	if err != nil {
		p.logger.Error("failed to get blocks for message", append(ctxlog.ContextFields(processingCtx), log.Error(err))...)
		p.postponeNotification(processingCtx, processingPhaseMetricName, postponementReasonFailedBlocksCollecting)
		return
	}
	return blocks, true
}

func (p *Processor) beginSendingTx(processingCtx processingContext) (tx database.NotificationTransaction, successful bool) {
	errGroup := errgroup.Group{}
	sendingTxChan := make(chan database.NotificationTransaction, 1)
	waitGroup := sync.WaitGroup{}
	waitGroup.Add(1)
	errGroup.Go(
		func() error {
			return p.beginSendingTransaction(processingCtx, &waitGroup, processingCtx.notification, sendingTxChan)
		},
	)
	waitGroup.Wait()

	err := p.commitProcessingTx(processingCtx)
	if err != nil {
		return
	}
	err = errGroup.Wait()
	if err != nil {
		return
	}
	return <-sendingTxChan, true
}

func (p *Processor) tryDeduplicate(processingCtx processingContext) bool {
	if p.config.DeduplicateByRecipient {
		shouldCancel, err := p.hasHigherPriorityNotification(processingCtx, processingCtx.notification)
		if err != nil {
			p.logger.Errorf("failed to deduplicate: %v", err)
			p.postponeNotification(processingCtx, processingPhaseMetricName, postponementReasonFailedDeduplication)
			return true
		}
		if shouldCancel {
			p.logger.Info(
				"notification will be cancelled due to deduplication",
				append(ctxlog.ContextFields(processingCtx), log.Error(err))...,
			)
			p.cancelNotification(processingCtx)
			return true
		}
	}
	return false
}

func (p *Processor) validateEmail(processingCtx processingContext, email string) bool {
	if len(email) == 0 {
		p.logger.Info("cancelled notification due to no email in order", ctxlog.ContextFields(processingCtx)...)
		p.cancelNotification(processingCtx)
		return false
	}
	return true
}

func (p *Processor) send(processingCtx processingContext, senderRequest sender.TransactionalRequest) {
	isSuccessful := false
	defer func() { p.writeSendingMetric(isSuccessful, processingCtx.notification.Subtype) }()
	p.requestSender(processingCtx, senderRequest)
	if p.updateNotificationStatus(processingCtx, models.NotificationStatusSent) != nil {
		return
	}
	isSuccessful = true
	p.logger.Info("notification has been sent successfully", ctxlog.ContextFields(processingCtx)...)
}

func (p *Processor) buildSenderRequest(
	renderedMessage *renderer.StructuredHTML,
	email string,
	processingCtx processingContext,
) sender.TransactionalRequest {
	return sender.TransactionalRequest{
		CampaignSlug: p.config.SenderCampaigns[processingCtx.notification.Subtype],
		ToEmail:      email,
		SendAsync:    false,
		Args: map[string]string{
			"head": renderedMessage.Head,
			"body": renderedMessage.Body,
		},
		Headers: map[string]string{"Subject": p.buildSubject(processingCtx)},
	}
}

func (p *Processor) buildSubject(processingCtx processingContext) string {
	textParams := map[string]interface{}{
		"dayNumber": processingCtx.order.StartDate.Day(),
		"monthName": constants.GenitiveMonthNames[processingCtx.order.StartDate.Month()],
	}
	settlementID, _ := p.routePointsExtractor.ExtractArrivalSettlementID(processingCtx.orderInfo)
	if accusativeWithPrep, ok := p.settlementAccusativeTitleExtractor.GetAccusativeTitleWithPreposition(settlementID); ok {
		textParams["settlementAccusativeWithPrep"] = accusativeWithPrep
	}
	subject, _ := tanker.TemplateToString(
		"pretrip_subject",
		subjects.GetSingular(processingCtx.notification.Subtype.String(), "ru"),
		textParams,
	)
	return subject
}

func (p *Processor) DebugSend(
	ctx context.Context,
	notification *models.Notification,
	email string,
) {
	ctx = ctxlog.WithFields(
		ctx,
		log.Int32("recipientID", *notification.RecipientID),
		log.UInt64("notificationID", notification.ID),
		log.String("orderID", *notification.OrderID),
	)
	orderInfo, err := p.orderProvider.GetOrderInfoByID(ctx, *notification.OrderID)
	if err != nil {
		p.logger.Error("failed to get order", append(ctxlog.ContextFields(ctx), log.Error(err))...)
		return
	}
	order, err := orderInfo.ToOrder()
	if err != nil {
		p.logger.Error("failed to parse order info", append(ctxlog.ContextFields(ctx), log.Error(err))...)
		return
	}
	blockByType, err := p.blocksProvider.GetBlocks(
		ctx,
		p.config.NotificationConfigByType[order.Type].Blocks,
		orderInfo,
		*notification,
	)
	if err != nil {
		p.logger.Error("failed to get blocks for message", append(ctxlog.ContextFields(ctx), log.Error(err))...)
		return
	}
	renderedMessage, err := p.renderer.RenderStructured(ctx, blockByType)
	if err != nil {
		p.logger.Error("failed to render message", append(ctxlog.ContextFields(ctx), log.Error(err))...)
		return
	}
	processingCtx := processingContext{
		Context:      ctx,
		notification: notification,
		orderInfo:    orderInfo,
		order:        order,
	}
	senderRequest := p.buildSenderRequest(renderedMessage, email, processingCtx)
	if _, err := p.senderClient.SendTransactional(ctx, senderRequest); err != nil {
		p.logger.Error("failed to send message", append(ctxlog.ContextFields(ctx), log.Error(err))...)
		return
	}
	p.logger.Info("notification has been sent successfully", ctxlog.ContextFields(ctx)...)
}

func (p *Processor) postponeNotification(processingCtx processingContext, phaseMetricName string, postponementReason string) {
	processingCtx.notification.NotifyAt = processingCtx.notification.NotifyAt.Add(p.config.PostponeInterval)
	if !processingCtx.notification.Deadline.IsZero() && processingCtx.notification.Deadline.Before(processingCtx.notification.NotifyAt) {
		p.handleNotificationWithExpiredDeadline(processingCtx, phaseMetricName)
		return
	}
	processingCtx.notification.Failures += 1
	processingCtx.notification.Status = models.NotificationStatusPostponed
	if err := processingCtx.tx.Update(*processingCtx.notification, p.logger); err != nil {
		p.logger.Error("failed to postpone notification", append(ctxlog.ContextFields(processingCtx), log.Error(err))...)
	} else {
		p.logger.Info("notification has been postponed", ctxlog.ContextFields(processingCtx)...)
		p.writePostponedNotificationsMetric(phaseMetricName, postponementReason, processingCtx.notification.Subtype)
	}
}

func (p *Processor) cancelNotification(processingCtx processingContext) {
	err := p.updateNotificationStatus(processingCtx, models.NotificationStatusCancelled)
	if err != nil {
		p.logger.Error("failed to cancel notification", append(ctxlog.ContextFields(processingCtx), log.Error(err))...)
	}
}

func (p *Processor) updateNotificationStatus(processingCtx processingContext, newStatus models.NotificationStatus) error {
	processingCtx.notification.Status = newStatus
	err := processingCtx.tx.Update(*processingCtx.notification, p.logger)
	if err != nil {
		p.logger.Error(
			"failed to update notification status",
			append(ctxlog.ContextFields(processingCtx), log.Error(err), log.Any("newStatus", newStatus))...,
		)
	} else {
		p.logger.Info(
			"notification status has been updated",
			append(ctxlog.ContextFields(processingCtx), log.Any("newStatus", newStatus))...,
		)
	}
	return err
}

func (p *Processor) isAlreadySent(ctx context.Context, orderID string, notificationType models.NotificationType, subtype models.NotificationSubtype) (bool, error) {
	alreadySentNotifications, err := p.notificationsRepository.AlreadySentForOrder(ctx, orderID, notificationType)
	if err != nil {
		return false, err
	}
	return isAlreadySentInternal(alreadySentNotifications, subtype), nil
}

func isAlreadySentInternal(alreadySentNotifications []models.Notification, subtype models.NotificationSubtype) bool {
	if subtype == models.NotificationAdhoc || subtype == models.NotificationWeekBefore {
		return len(alreadySentNotifications) > 0
	}
	for _, notification := range alreadySentNotifications {
		if notification.Subtype == models.NotificationDayBefore || notification.Subtype == models.NotificationAdhoc {
			return true
		}
	}
	return len(alreadySentNotifications) > 1
}

func (p *Processor) isMultiOrderCorrelationID(ctx context.Context, order models.Order) (bool, error) {
	if order.Type != models.OrderTrain || order.CorrelationID == "" {
		return false, nil
	}
	// fetch orders with the same correlation ID
	relatedOrders, err := p.ordersRepository.GetByCorrelationID(ctx, order.CorrelationID)
	if err != nil {
		return false, err
	}
	return len(relatedOrders) >= 2, nil
}

func (p *Processor) hasHigherPriorityNotification(ctx context.Context, notification *models.Notification) (bool, error) {
	var higherPriorityTypes map[models.OrderType]void
	if notification.Order == nil {
		return false, ErrNotificationWithNoOrder
	}
	switch notification.Order.Type {
	case models.OrderHotel:
		higherPriorityTypes = toMap(
			[]models.OrderType{
				models.OrderTrain,
			},
		)
	}
	if len(higherPriorityTypes) == 0 {
		return false, nil
	}
	today := p.clock.Now()
	weekAgo := today.Add(-7 * 24 * time.Hour)
	nextWeek := today.Add(7 * 24 * time.Hour)
	pastOrders, err := p.notificationsRepository.GetForRecipient(ctx, *notification.RecipientID, weekAgo, today)
	if err != nil {
		return false, err
	}
	hasBeenSent := func(notification models.Notification) bool {
		if !isPretrip(&notification) {
			return false
		}
		_, hasHigherPriority := higherPriorityTypes[notification.Order.Type]
		return hasHigherPriority && notification.Status == models.NotificationStatusSent
	}
	if any(pastOrders, hasBeenSent) {
		return true, nil
	}
	futureOrders, err := p.notificationsRepository.GetForRecipient(ctx, *notification.RecipientID, today, nextWeek)
	if err != nil {
		return false, err
	}
	notCancelled := func(notification models.Notification) bool {
		if !isPretrip(&notification) {
			return false
		}
		_, hasHigherPriority := higherPriorityTypes[notification.Order.Type]
		return hasHigherPriority && notification.Status != models.NotificationStatusCancelled
	}
	if any(futureOrders, notCancelled) {
		return true, nil
	}
	return false, nil
}

func isPretrip(notification *models.Notification) bool {
	return notification.Type.Name == models.NotificationTypePretrip.Name
}

func (p *Processor) writeSendingMetric(isSuccessful bool, subtype models.NotificationSubtype) {
	metricName := successfulSendingMetricName
	if !isSuccessful {
		metricName = failedSendingMetricName
	}
	metrics.GlobalAppMetrics().GetOrCreateCounter(
		pretripMetricsPrefix,
		map[string]string{"phase": sendingPhaseMetricName, "subtype": subtype.String()},
		metricName,
	).Inc()
}

func (p *Processor) needToNotify(processingCtx processingContext) bool {
	processingCtx.Context = ctxlog.WithFields(
		processingCtx,
		log.String("orderState", string(processingCtx.order.State)),
		log.String("orderType", processingCtx.order.Type.String()),
	)

	//notification := processingCtx.notification
	if !slices.ContainsString(p.config.EnabledOrderTypes, processingCtx.order.Type.String()) {
		p.logger.Info("pretrip is not enabled for order type", ctxlog.ContextFields(processingCtx)...)
		return false
	}

	// https://st.yandex-team.ru/RASPTICKETS-20210
	if processingCtx.notification.Subtype != models.NotificationDayBefore && !p.rollOutService.IsEnabledForEmail(processingCtx.notification.Recipient.GetEmail()) {
		p.logger.Warn("cancelled notification due to roll out policy", ctxlog.ContextFields(processingCtx)...)
		p.cancelNotification(processingCtx)
		return false
	}

	if !processingCtx.notification.Deadline.IsZero() && processingCtx.notification.Deadline.Before(p.clock.Now()) {
		p.logger.Info("notification's deadline has already passed", ctxlog.ContextFields(processingCtx)...)
		p.cancelNotification(processingCtx)
		return false
	}

	settlementID, err := p.routePointsExtractor.ExtractArrivalSettlementID(processingCtx.orderInfo)
	if err != nil {
		p.logger.Warn(
			"cancelled notification due to unable to impossibility to extract destination settlement",
			ctxlog.ContextFields(processingCtx)...,
		)
		p.cancelNotification(processingCtx)
		return false
	}

	if !p.settlementIDsWhitelist.Contains(settlementID) {
		p.logger.Info(
			"cancelled notification due to settlement destination is not in whitelist",
			append(ctxlog.ContextFields(processingCtx), log.Int("settlementID", settlementID))...,
		)
		p.cancelNotification(processingCtx)
		return false
	}

	if len(processingCtx.orderInfo.TrainOrderItems) > 1 {
		p.logger.Warn(
			"cancelled notification: order has more than one train order item",
			append(ctxlog.ContextFields(processingCtx), log.Int("trainOrderItemsCount", len(processingCtx.orderInfo.TrainOrderItems)))...,
		)
		p.cancelNotification(processingCtx)
		return false
	}

	if !processingCtx.order.IsNotifyable() {
		p.cancelNotification(processingCtx)
		p.logger.Info("cancelled notification due to order being no longer fulfilled", ctxlog.ContextFields(processingCtx)...)
		return false
	}

	if isCorrect := p.checkIsAlreadySent(processingCtx); !isCorrect {
		return false
	}

	if isCorrect := p.checkIsMultiOrder(processingCtx); !isCorrect {
		return false
	}

	return true
}

func (p *Processor) checkIsMultiOrder(processingCtx processingContext) (isCorrect bool) {
	isMultiOrder, err := p.isMultiOrderCorrelationID(processingCtx, processingCtx.order)
	if err != nil {
		p.logger.Error(
			"failed to test if there are orders linked by the correlation ID",
			append(ctxlog.ContextFields(processingCtx), log.Error(err))...,
		)
		p.postponeNotification(processingCtx, processingPhaseMetricName, postponementReasonFailedAlreadySentCheck)
		return false
	}
	if isMultiOrder {
		p.logger.Info("cancelled notification since there's a related by correlation ID order", ctxlog.ContextFields(processingCtx)...)
		p.cancelNotification(processingCtx)
		return false
	}
	return true
}

func (p *Processor) checkIsAlreadySent(processingCtx processingContext) (isCorrect bool) {
	alreadySent, err := p.isAlreadySent(processingCtx, processingCtx.order.ID, processingCtx.notification.Type, processingCtx.notification.Subtype)
	if err != nil {
		p.logger.Error("failed to test already-sent case for the message", append(ctxlog.ContextFields(processingCtx), log.Error(err))...)
		p.postponeNotification(processingCtx, processingPhaseMetricName, postponementReasonFailedAlreadySentCheck)
		return false
	}
	if alreadySent {
		p.cancelNotification(processingCtx)
		p.logger.Info(
			"cancelled notification since another one has been already sent for this order",
			ctxlog.ContextFields(processingCtx)...,
		)
		return false
	}
	return true
}

func (p *Processor) writePostponedNotificationsMetric(phaseMetricName string, reason string, subtype models.NotificationSubtype) {
	metrics.GlobalAppMetrics().GetOrCreateCounter(
		pretripMetricsPrefix,
		map[string]string{"phase": phaseMetricName, "reason": reason, "subtype": subtype.String()},
		postponedNotificationMetricName,
	).Inc()
}

func (p *Processor) requestSender(ctx context.Context, request sender.TransactionalRequest) {
	startSendingAt := time.Now()
	var err error
	defer func() {
		tags := map[string]string{"is_error": strconv.FormatBool(err != nil)}
		metrics.GlobalAppMetrics().GetOrCreateCounter(
			senderMetricsPrefix,
			tags,
			senderRequestsMetricName,
		).Inc()
		metrics.GlobalAppMetrics().GetOrCreateHistogram(
			senderMetricsPrefix,
			tags,
			senderTimingsMetricName,
			senderLatencyBuckets,
		).RecordDuration(time.Since(startSendingAt))
	}()

	if _, err = p.senderClient.SendTransactional(ctx, request); err != nil {
		p.logger.Error(
			"failed to send message",
			append(ctxlog.ContextFields(ctx), log.Error(err), log.Float64("duration_seconds", time.Since(startSendingAt).Seconds()))...,
		)
		return
	}
}

func (p *Processor) handleNotificationWithExpiredDeadline(processingCtx processingContext, phaseMetricName string) {
	metrics.GlobalAppMetrics().GetOrCreateCounter(
		pretripMetricsPrefix,
		map[string]string{"phase": phaseMetricName, "subtype": processingCtx.notification.Subtype.String()},
		expiredNotificationMetricName,
	).Inc()
	p.logger.Info(
		"notification should be cancelled due to expired deadline",
		append(ctxlog.ContextFields(processingCtx), log.Time("notificationDeadline", processingCtx.notification.Deadline))...,
	)
	p.cancelNotification(processingCtx)
	if err := processingCtx.tx.Commit(p.logger); err != nil {
		p.logger.Error(
			"failed to cancel notification with expired deadline",
			append(ctxlog.ContextFields(processingCtx), log.Error(err))...,
		)
		return
	}
	err := p.expiredNotificationDeadlineHandler.OnNotificationDeadline(
		processingCtx,
		processingCtx.notification,
		processingCtx.orderInfo,
	)
	if err != nil {
		p.logger.Error(
			"failed to handle notification with expired deadline",
			append(ctxlog.ContextFields(processingCtx), log.Error(err))...,
		)
	}
}

func (p *Processor) render(processingCtx processingContext, blocks []renderer.Block) (html *renderer.StructuredHTML, successful bool) {
	renderedMessage, err := p.renderer.RenderStructured(processingCtx, blocks)
	go func() {
		blocksJSON, _ := json.Marshal(blocks)
		err := p.renderLogger.Log(
			&record.RenderLogRecord{
				Timestamp:      time.Now().Unix(),
				IterationId:    processingCtx.iterationID.String(),
				OrderId:        processingCtx.order.ID,
				NotificationId: int64(processingCtx.notification.ID),
				BlocksJson:     string(blocksJSON),
				RenderedHtml:   formatHTML(renderedMessage),
			},
		)
		if err != nil {
			p.logger.Error("failed to write pretripRenderLog record", append(ctxlog.ContextFields(processingCtx), log.Error(err))...)
		}
	}()

	if err == nil {
		p.logger.Info("rendered successfully", ctxlog.ContextFields(processingCtx)...)
		return renderedMessage, true
	}
	p.logger.Error("failed to render message", append(ctxlog.ContextFields(processingCtx), log.Error(err))...)
	p.postponeNotification(processingCtx, processingPhaseMetricName, postponementReasonFailedRendering)
	return
}

func (p *Processor) commitProcessingTx(processingCtx processingContext) error {
	err := processingCtx.tx.Commit(p.logger)
	if err == nil {
		return nil
	}
	p.logger.Error("failed to commit before mailing", append(ctxlog.ContextFields(processingCtx), log.Error(err))...)
	tx, err := p.notificationsRepository.BeginTransaction(processingCtx, *processingCtx.notification, p.txOptions)
	if err != nil {
		p.logger.Error("failed to begin postpone transaction", append(ctxlog.ContextFields(processingCtx), log.Error(err))...)
		return err
	}
	p.postponeNotification(processingCtx, processingPhaseMetricName, postponementReasonFailedTxCommit)
	p.logger.Info("transaction is going to be committed", append(ctxlog.ContextFields(processingCtx), log.Error(err))...)
	if err := tx.Commit(p.logger); err != nil {
		p.logger.Error("failed to commit postpone transaction", append(ctxlog.ContextFields(processingCtx), log.Error(err))...)
		return err
	}
	return nil
}

func (p *Processor) beginSendingTransaction(
	ctx context.Context,
	waitGroup *sync.WaitGroup,
	notification *models.Notification,
	sendingTxChan chan<- database.NotificationTransaction,
) error {
	waitGroup.Done()
	sendingTx, err := p.notificationsRepository.BeginTransaction(ctx, *notification, p.txOptions)
	if err != nil {
		p.logger.Error("failed to begin sending transaction for notification", append(ctxlog.ContextFields(ctx), log.Error(err))...)
	}
	sendingTxChan <- sendingTx
	return err
}

func (p *Processor) prepareProcessingContext(
	ctx context.Context,
	notification *models.Notification,
	processingTx database.NotificationTransaction,
	iterationID uuid.UUID,
) (processingCtx processingContext, successful bool) {
	ctx = ctxlog.WithFields(
		ctx,
		log.Int32("recipientID", *notification.RecipientID),
		log.UInt64("notificationID", notification.ID),
		log.String("notificationSubtype", notification.Subtype.String()),
		log.String("orderID", notification.Order.ID),
		log.String("iterationID", iterationID.String()),
	)
	processingCtx = processingContext{
		Context:      ctx,
		notification: notification,
		tx:           processingTx,
		iterationID:  iterationID,
	}
	localOrder := notification.Order
	if localOrder == nil {
		p.logger.Error("notification doesn't have related order and will be cancelled", ctxlog.ContextFields(ctx)...)
		p.cancelNotification(processingCtx)
		return
	}
	var err error
	processingCtx.orderInfo, err = p.orderProvider.GetOrderInfoByID(ctx, localOrder.ID)
	if err != nil {
		p.logger.Error("failed to get order", append(ctxlog.ContextFields(ctx), log.Error(err))...)
		p.postponeNotification(processingCtx, processingPhaseMetricName, postponementReasonFailedOrderFetching)
		return
	}
	p.logger.Info("got order info", ctxlog.ContextFields(ctx)...)
	if !p.validateEmail(processingCtx, processingCtx.orderInfo.GetEmail()) {
		return
	}
	if p.tryDeduplicate(processingCtx) {
		return
	}
	processingCtx.order, err = processingCtx.orderInfo.ToOrder()
	if err != nil {
		p.logger.Error("failed to parse order info", append(ctxlog.ContextFields(ctx), log.Error(err))...)
		p.postponeNotification(processingCtx, processingPhaseMetricName, postponementReasonFailedOrderMapping)
		return
	}
	if !p.needToNotify(processingCtx) {
		return
	}
	return processingCtx, true
}

func formatHTML(structuredHTML *renderer.StructuredHTML) string {
	if structuredHTML == nil {
		return ""
	}
	return fmt.Sprintf(
		"<!doctype html>"+
			"\n<html xmlns=\"http://www.w3.org/1999/xhtml\" xmlns:v=\"urn:schemas-microsoft-com:vml\" "+
			"xmlns:o=\"urn:schemas-microsoft-com:office:office\">\n"+
			"<head>%s</head>\n<body>%s</body>\n</html>",
		structuredHTML.Head,
		structuredHTML.Body,
	)
}

func any(notifications []models.Notification, predicate func(notification models.Notification) bool) bool {
	for _, notification := range notifications {
		if predicate(notification) {
			return true
		}
	}
	return false
}

func toMap(typeNames []models.OrderType) map[models.OrderType]void {
	result := make(map[models.OrderType]void)
	for _, typeName := range typeNames {
		result[typeName] = void{}
	}
	return result
}
