package scheduling

import (
	"time"

	"a.yandex-team.ru/travel/library/go/containers"
	"a.yandex-team.ru/travel/notifier/internal/models"
)

var (
	plannedTripThreshold     = 8 * 24 * time.Hour
	minimalPlanningThreshold = 3 * time.Hour
)

type NotificationsBuilderConfig struct {
	IsTesting                 bool
	TestingEmails             containers.Set[string]
	AdhocSendingInterval      time.Duration
	WeekBeforeSendingInterval time.Duration
	DayBeforeSendingInterval  time.Duration
}

type NotificationsBuilder struct {
	config NotificationsBuilderConfig
}

func NewNotificationsBuilder(config NotificationsBuilderConfig) NotificationsBuilder {
	return NotificationsBuilder{config: config}
}

func (b *NotificationsBuilder) Build(order models.Order, recipient models.Recipient, now time.Time) ([]models.Notification, error) {
	// no notifications for cancelled/unpaid orders
	if !order.IsNotifyable() || order.Type == models.OrderAvia {
		return []models.Notification{}, nil
	}
	// trip starts in less than three hours - no time to notify
	if order.StartDate.Before(now.Add(minimalPlanningThreshold)) {
		return []models.Notification{}, nil
	}

	adhocNotifyAt := b.calcAdhocNotifyAt(now, recipient)
	if b.isAdhoc(order, now) {
		return []models.Notification{
			b.buildNotification(
				adhocNotifyAt,
				b.calcAdhocDeadline(adhocNotifyAt, order.StartDate, recipient),
				recipient,
				models.NotificationAdhoc,
				order,
			),
		}, nil
	}
	weekBeforeNotifyAt := b.calcWeekBeforeNotifyAt(now, order, recipient)
	weekBefore := b.buildNotification(
		weekBeforeNotifyAt,
		b.calcWeekBeforeDeadline(weekBeforeNotifyAt, recipient),
		recipient,
		models.NotificationWeekBefore,
		order,
	)
	dayBeforeNotifyAt := b.calcDayBeforeNotifyAt(now, order, recipient)
	dayBefore := b.buildNotification(
		dayBeforeNotifyAt,
		b.calcDayBeforeDeadline(dayBeforeNotifyAt, order.StartDate, recipient),
		recipient,
		models.NotificationDayBefore,
		order,
	)
	return []models.Notification{weekBefore, dayBefore}, nil
}

func (b *NotificationsBuilder) isAdhoc(order models.Order, now time.Time) bool {
	return order.StartDate.Before(now.Add(plannedTripThreshold))
}

func (b *NotificationsBuilder) buildNotification(
	notifyAt time.Time,
	deadline time.Time,
	recipient models.Recipient,
	subtype models.NotificationSubtype,
	order models.Order,
) models.Notification {
	return models.NewNotification(
		notifyAt,
		deadline,
		models.NotificationStatusPlanned,
		models.NotificationTypePretrip,
		models.NotificationChannelEmail,
		models.DispatchTypePush,
	).WithSubtype(subtype).WithOrder(order).WithRecipient(recipient)
}

func (b *NotificationsBuilder) calcAdhocNotifyAt(now time.Time, recipient models.Recipient) time.Time {
	if b.config.IsTesting && b.config.TestingEmails.Contains(recipient.GetEmail()) {
		return now.Add(b.config.AdhocSendingInterval)
	}
	return now.Add(minimalPlanningThreshold)
}

func (b *NotificationsBuilder) calcWeekBeforeNotifyAt(now time.Time, order models.Order, recipient models.Recipient) time.Time {
	if b.config.IsTesting && b.config.TestingEmails.Contains(recipient.GetEmail()) {
		return now.Add(b.config.WeekBeforeSendingInterval)
	}
	return order.StartDate.AddDate(0, 0, -7)
}

func (b *NotificationsBuilder) calcDayBeforeNotifyAt(now time.Time, order models.Order, recipient models.Recipient) time.Time {
	if b.config.IsTesting && b.config.TestingEmails.Contains(recipient.GetEmail()) {
		return now.Add(b.config.DayBeforeSendingInterval)
	}
	return order.StartDate.AddDate(0, 0, -1)
}

func (b *NotificationsBuilder) calcAdhocDeadline(notifyAt time.Time, startDate time.Time, recipient models.Recipient) time.Time {
	if b.config.IsTesting && b.config.TestingEmails.Contains(recipient.GetEmail()) {
		return notifyAt.Add(2 * time.Minute)
	}
	return startDate.Add(-4 * time.Hour)
}

func (b *NotificationsBuilder) calcDayBeforeDeadline(notifyAt time.Time, startDate time.Time, recipient models.Recipient) time.Time {
	if b.config.IsTesting && b.config.TestingEmails.Contains(recipient.GetEmail()) {
		return notifyAt.Add(2 * time.Minute)
	}
	return startDate.Add(-4 * time.Hour)
}

func (b *NotificationsBuilder) calcWeekBeforeDeadline(notifyAt time.Time, recipient models.Recipient) time.Time {
	if b.config.IsTesting && b.config.TestingEmails.Contains(recipient.GetEmail()) {
		return notifyAt.Add(2 * time.Minute)
	}
	return notifyAt.Add(24 * time.Hour)
}
