package pretrip

import (
	"context"
	"fmt"

	"github.com/jonboulle/clockwork"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/slices"
	"a.yandex-team.ru/travel/library/go/containers"
	"a.yandex-team.ru/travel/library/go/metrics"
	"a.yandex-team.ru/travel/notifier/internal/models"
	"a.yandex-team.ru/travel/notifier/internal/orderchanges"
	"a.yandex-team.ru/travel/notifier/internal/orders"
	"a.yandex-team.ru/travel/notifier/internal/service/pretrip/interfaces"
	"a.yandex-team.ru/travel/notifier/internal/service/pretrip/scheduling"
)

type NotificationsService struct {
	logger                 log.Logger
	notificationsScheduler interfaces.NotificationsScheduler
	notificationBuilder    scheduling.NotificationsBuilder
	recipientsRepository   interfaces.RecipientsRepository
	routePointsExtractor   interfaces.RoutePointsExtractor
	pretripConfig          PlanningConfig
	clock                  clockwork.Clock
	settlementIDsWhitelist containers.Set[int]
	rollOutService         interfaces.RollOutService
	emailEnabler           interfaces.EmailEnabler
	settlementDataProvider interfaces.SettlementDataProvider
	stationDataProvider    interfaces.StationDataProvider
}

func NewPretripNotificationsService(
	logger log.Logger,
	notificationsScheduler interfaces.NotificationsScheduler,
	notificationBuilder scheduling.NotificationsBuilder,
	recipientsRepository interfaces.RecipientsRepository,
	routePointsExtractor interfaces.RoutePointsExtractor,
	pretripConfig PlanningConfig,
	clock clockwork.Clock,
	rollOutService interfaces.RollOutService,
	emailEnabler interfaces.EmailEnabler,
	settlementDataProvider interfaces.SettlementDataProvider,
	stationDataProvider interfaces.StationDataProvider,
) *NotificationsService {
	return &NotificationsService{
		logger:                 logger.WithName("PretripNotificationsService"),
		notificationsScheduler: notificationsScheduler,
		notificationBuilder:    notificationBuilder,
		recipientsRepository:   recipientsRepository,
		routePointsExtractor:   routePointsExtractor,
		pretripConfig:          pretripConfig,
		clock:                  clock,
		settlementIDsWhitelist: containers.SetOf(pretripConfig.SettlementsWhitelist...),
		rollOutService:         rollOutService,
		emailEnabler:           emailEnabler,
		settlementDataProvider: settlementDataProvider,
		stationDataProvider:    stationDataProvider,
	}
}

func (p *NotificationsService) OnOrderChanged(
	ctx context.Context,
	orderInfo *orders.OrderInfo,
) (result orderchanges.OrderChangedResult) {
	defer func() {
		if r := recover(); r != nil {
			result.Code = "panic"
		}
		if result.Code != "" {
			metrics.GlobalAppMetrics().GetOrCreateCounter(
				pretripMetricsPrefix,
				map[string]string{"code": result.Code},
				notPlannedNotificationMetricName,
			).Inc()
		}
		p.logger.Info(
			"OnOrderChanged request processed",
			log.String("orderID", orderInfo.ID),
			log.String("code", result.Code),
			log.String("message", result.Message),
		)
	}()
	if err := p.validateArrivalSettlement(orderInfo); err != nil {
		return orderchanges.OrderChangedResult{
			Status:  orderchanges.OrderChangedStatusOK,
			Message: fmt.Sprintf("notifications haven't been planned for order %s because: %v", orderInfo.ID, err),
			Code:    "invalid_destination",
		}
	}

	if err := p.validateRouteDistance(orderInfo); err != nil {
		return orderchanges.OrderChangedResult{
			Status:  orderchanges.OrderChangedStatusOK,
			Message: fmt.Sprintf("notifications haven't been planned for order %s because: %v", orderInfo.ID, err),
			Code:    "short_route",
		}
	}

	_, err := p.notificationsScheduler.CancelPlannedNotificationsByOrderID(ctx, orderInfo.ID, models.NotificationTypePretrip, models.DispatchTypePush)
	if err != nil {
		return orderchanges.OrderChangedResult{
			Status:  orderchanges.OrderChangedStatusTemporaryError,
			Message: fmt.Sprintf("failed to cancel notifications for order %s: %s", orderInfo.ID, err),
			Code:    "cancel_planned_error",
		}
	}

	// nowhere to send notifications
	email := orderInfo.GetEmail()

	if len(email) == 0 {
		p.logger.Warn("empty email", log.String("orderID", orderInfo.ID))
		return orderchanges.OrderChangedResult{Status: orderchanges.OrderChangedStatusOK, Code: "empty_email"}
	}

	alreadySentNotifications, err := p.notificationsScheduler.AlreadySentForOrder(ctx, orderInfo.ID)
	if err != nil {
		return orderchanges.OrderChangedResult{
			Status:  orderchanges.OrderChangedStatusTemporaryError,
			Message: fmt.Sprintf("failed to verify whether notifications have been sent already for order %s: %s", orderInfo.ID, err),
			Code:    "already_sent_error",
		}
	}
	if isAlreadySent(alreadySentNotifications) {
		p.logger.Info("Won't plan new notifications for order due to it has sent notifications")
		return orderchanges.OrderChangedResult{Status: orderchanges.OrderChangedStatusOK, Code: "already_sent"}
	}

	recipient, err := p.recipientsRepository.GetOrCreate(ctx, p.buildRecipient(email))
	if err != nil {
		p.logger.Error("failed to create a recipient", log.Error(err), log.String("orderID", orderInfo.ID))
		return orderchanges.OrderChangedResult{
			Status:  orderchanges.OrderChangedStatusFailure,
			Message: fmt.Sprintf("failed to create a recipient: %s", err),
			Code:    "recipient_create_error",
		}
	}

	// https://st.yandex-team.ru/RASPTICKETS-20210
	if !p.emailEnabler.Allows(recipient.GetEmail()) && !p.rollOutService.IsEnabledForEmail(recipient.GetEmail()) {
		message := "pretrip functionality isn't enabled for recipient. notifications haven't been scheduled."
		p.logger.Info(message, log.Int32("recipientID", recipient.ID), log.String("orderID", orderInfo.ID))
		return orderchanges.OrderChangedResult{Status: orderchanges.OrderChangedStatusOK, Message: message, Code: "rollout_filter"}
	}

	order, _ := orderInfo.ToOrder()
	if !slices.ContainsString(p.pretripConfig.EnabledOrderTypes, order.Type.String()) {
		p.logger.Info(
			"pretrip is not enabled for for order type",
			log.String("orderType", order.Type.String()),
			log.String("orderID", order.ID),
		)
		return orderchanges.OrderChangedResult{
			Status:  orderchanges.OrderChangedStatusOK,
			Message: fmt.Sprintf("pretrip is not enabled for for order type: %s", order.Type),
			Code:    "not_enabled_order_type",
		}
	}

	settlementID, err := p.routePointsExtractor.ExtractArrivalSettlementID(orderInfo)
	if err == nil {
		order.DestinationSettlementID = settlementID
	} else {
		p.logger.Error("failed to extract destination settlement ID", log.Error(err), log.String("orderID", order.ID))
		return orderchanges.OrderChangedResult{
			Status:  orderchanges.OrderChangedStatusOK,
			Message: fmt.Sprintf("failed to extract destination settlement ID: %s", err),
			Code:    "destination_extract_error",
		}
	}

	notifications, err := p.notificationBuilder.Build(order, *recipient, p.clock.Now())
	if err != nil {
		p.logger.Error("failed to build notifications", log.Error(err), log.String("orderID", order.ID))
		return orderchanges.OrderChangedResult{
			Status:  orderchanges.OrderChangedStatusFailure,
			Message: fmt.Sprintf("couldn't build notification: %s", err),
			Code:    "build_error",
		}
	}
	if len(notifications) == 0 {
		p.logger.Info("0 notifications have been built for order", log.String("orderID", order.ID))
		return orderchanges.OrderChangedResult{Status: orderchanges.OrderChangedStatusOK}
	}
	if _, err := p.notificationsScheduler.ScheduleMany(ctx, notifications, p.clock.Now()); err != nil {
		p.logger.Error("failed to schedule notifications", log.Error(err), log.String("orderID", order.ID))
		return orderchanges.OrderChangedResult{
			Status:  orderchanges.OrderChangedStatusTemporaryError,
			Message: fmt.Sprintf("couldn't schedule notification: %s", err),
			Code:    "schedule_error",
		}
	}
	p.logger.Info(
		"notifications have been successfully scheduled",
		log.String("orderID", order.ID),
		log.Int("notificationsCount", len(notifications)),
	)
	for _, notification := range notifications {
		metrics.GlobalAppMetrics().GetOrCreateCounter(
			pretripMetricsPrefix,
			map[string]string{"subtype": notification.Subtype.String()},
			plannedNotificationMetricName,
		).Inc()
	}
	return orderchanges.OrderChangedResult{Status: orderchanges.OrderChangedStatusOK}
}

func (p *NotificationsService) buildRecipient(email string) models.Recipient {
	return models.NewRecipient().WithEmail(email)
}

func isAlreadySent(notifications []models.Notification) bool {
	if len(notifications) > 1 {
		return true
	}
	if len(notifications) < 1 {
		return false
	}
	return notifications[0].Subtype == models.NotificationAdhoc || notifications[0].Subtype == models.NotificationDayBefore
}

func (p *NotificationsService) validateArrivalSettlement(orderInfo *orders.OrderInfo) error {
	settlementID, err := p.routePointsExtractor.ExtractArrivalSettlementID(orderInfo)
	if err != nil {
		p.logger.Warn(
			"notifications won't be planned due to arrival settlement isn't correct",
			log.Error(err),
			log.String("orderID", orderInfo.ID),
			log.String("orderType", orderInfo.Type.String()),
		)
		return fmt.Errorf("couldn't extract settlementID from order: %w", err)
	}
	if !p.settlementIDsWhitelist.Contains(settlementID) {
		p.logger.Warn(
			"notifications won't be planned due to arrival settlement isn't in whitelist",
			log.String("orderID", orderInfo.ID),
		)
		return fmt.Errorf("settlementID %d not in whitelist", settlementID)
	}
	return nil
}

func (p *NotificationsService) validateRouteDistance(orderInfo *orders.OrderInfo) error {
	if orderInfo.Type != orders.OrderTypeTrain {
		return nil
	}
	arrivalStationID, _ := p.routePointsExtractor.ExtractArrivalStationID(orderInfo)
	departureStationID, err := p.routePointsExtractor.ExtractDepartureStationID(orderInfo)
	if err != nil {
		return fmt.Errorf("couldn't extract departure station id: %w", err)
	}
	distance, found := p.stationDataProvider.GetDistance(departureStationID, arrivalStationID)
	if !found {
		arrivalSettlementID, _ := p.routePointsExtractor.ExtractArrivalSettlementID(orderInfo)
		departureSettlementID, err := p.routePointsExtractor.ExtractDepartureSettlementID(orderInfo)
		if err != nil {
			return fmt.Errorf("couldn't extract departure settlement id: %w", err)
		}
		distance, found = p.settlementDataProvider.GetDistance(departureSettlementID, arrivalSettlementID)
	}
	if !found || distance >= p.pretripConfig.TrainMinDistanceKM {
		return nil
	}
	return fmt.Errorf(
		"distance between stations %d and %d is less than %f",
		departureStationID,
		arrivalStationID,
		p.pretripConfig.TrainMinDistanceKM,
	)
}
