package processing

import (
	"context"
	"fmt"
	"sync"
	"time"

	"github.com/gofrs/uuid"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/library/go/metrics"
	"a.yandex-team.ru/travel/notifier/internal/database"
	"a.yandex-team.ru/travel/notifier/internal/models"
)

type NotificationProcessor interface {
	Process(
		ctx context.Context,
		notification *models.Notification,
		notificationTx database.NotificationTransaction,
		iterationID uuid.UUID,
	)
}

type NotificationsRepository interface {
	BeginTransaction(
		ctx context.Context,
		notification models.Notification,
		options database.TransactionOptions,
	) (database.NotificationTransaction, error)
	GetForRecipient(ctx context.Context, recipientID int32, from time.Time, until time.Time) ([]models.Notification, error)
	AlreadySentForOrder(context.Context, string, models.NotificationType) ([]models.Notification, error)
}

type Service struct {
	processorByType         map[string]NotificationProcessor
	notificationsRepository NotificationsRepository
	logger                  log.Logger
	config                  Config
}

type Option func(*Service)

func WithProcessor(notificationType models.NotificationType, processor NotificationProcessor) Option {
	return func(service *Service) {
		service.processorByType[notificationType.Name] = processor
	}
}

func NewService(logger log.Logger, notificationsRepository NotificationsRepository, options ...Option) *Service {
	service := &Service{
		logger:                  logger.WithName("RootProcessingService"),
		notificationsRepository: notificationsRepository,
		processorByType:         make(map[string]NotificationProcessor),
	}
	for _, option := range options {
		option(service)
	}
	return service
}

func (s *Service) Process(ctx context.Context, notifications []models.Notification, iterationID uuid.UUID) (waitAllProcessed func()) {
	s.logger.Info(
		"start processing notifications",
		log.Int("count", len(notifications)),
		log.String("iterationID", iterationID.String()),
		log.UInt64s("notificationIDs", toIDs(notifications)),
	)
	allAreBeingProcessed := &sync.WaitGroup{}
	allAreBeingProcessed.Add(len(notifications))
	allProcessed := &sync.WaitGroup{}
	allProcessed.Add(len(notifications))
	once := sync.Once{}
	waitAllProcessed = func() {
		once.Do(allProcessed.Wait)
	}
	notificationsByStatus := make(map[models.NotificationStatus][]*models.Notification)
	notificationsByStatusLock := &sync.Mutex{}
	for _, notification := range notifications {
		localNotification := notification
		go s.lockAndProcess(
			ctx,
			iterationID,
			localNotification,
			allAreBeingProcessed,
			allProcessed,
			notificationsByStatus,
			notificationsByStatusLock,
		)
	}
	allAreBeingProcessed.Wait()
	go s.logResults(iterationID, waitAllProcessed, notificationsByStatus)
	return waitAllProcessed
}

func toIDs(notifications []models.Notification) []uint64 {
	result := make([]uint64, 0, len(notifications))
	for _, notification := range notifications {
		result = append(result, notification.ID)
	}
	return result
}

func (s *Service) getTransaction(
	ctx context.Context,
	iterationID uuid.UUID,
	notification models.Notification,
	allAreBeingProcessed, allProcessed *sync.WaitGroup,
) (tx database.NotificationTransaction, err error) {
	defer func() {
		if r := recover(); r != nil {
			allAreBeingProcessed.Done()
			allProcessed.Done()
			err = fmt.Errorf("panic happened during transaction begin: %v", err)
			s.logger.Error("panic happened during transaction begin", log.Any("panic", r))
			return
		}
		allAreBeingProcessed.Done()
		if err != nil {
			s.logger.Error(
				"failed to begin processing transaction for notification",
				log.UInt64("notificationID", notification.ID),
				log.Error(err),
				log.String("iterationID", iterationID.String()),
			)
			allProcessed.Done()
		}
	}()

	return s.notificationsRepository.BeginTransaction(
		ctx, notification, database.TransactionOptions{
			StatementTimeout:                s.config.DBStatementTimeout,
			LockTimeout:                     s.config.DBLockTimeout,
			IdleInTransactionSessionTimeout: s.config.DBIdleInTransactionTimeout,
		},
	)
}

func (s *Service) lockAndProcess(
	ctx context.Context,
	iterationID uuid.UUID,
	notification models.Notification,
	allAreBeingProcessed *sync.WaitGroup,
	allProcessed *sync.WaitGroup,
	notificationsByStatus map[models.NotificationStatus][]*models.Notification,
	notificationsByStatusLock *sync.Mutex,
) {
	tx, err := s.getTransaction(ctx, iterationID, notification, allAreBeingProcessed, allProcessed)
	if err != nil {
		return
	}
	processingStartTime := time.Now()
	s.logger.Info(
		"	transaction has been begun",
		log.UInt64("notificationID", notification.ID),
		log.String("iterationID", iterationID.String()),
	)
	notificationProcessed := sync.WaitGroup{}
	notificationProcessed.Add(1)
	go func() {
		defer func() {
			if r := recover(); r != nil {
				s.logger.Error("panic happened during transaction commit", log.Any("panic", r))
			}
			notificationProcessed.Done()
			s.logger.Info(
				"notification has been processed",
				log.UInt64("notificationID", notification.ID),
				log.String("iterationID", iterationID.String()),
			)
			allProcessed.Done()
			sendMetrics(processingStartTime, notification)
		}()
		s.processSingle(ctx, &notification, tx, iterationID)
		s.logger.Info(
			"transaction is going to be committed",
			log.UInt64("notificationID", notification.ID),
			log.String("iterationID", iterationID.String()),
		)
		if err := tx.Commit(s.logger); err != nil && err != database.ErrTxClosed {
			s.logger.Error(
				"failed to commit transaction",
				log.UInt64("notificationID", notification.ID),
				log.Error(err),
				log.String("iterationID", iterationID.String()),
			)
			if err := tx.Rollback(s.logger); err != nil && err != database.ErrTxClosed {
				s.logger.Error(
					"failed to rollback transaction",
					log.UInt64("notificationID", notification.ID),
					log.Error(err),
					log.String("iterationID", iterationID.String()),
				)
			}
		}
	}()
	go func() {
		notificationProcessed.Wait()
		notificationsByStatusLock.Lock()
		defer notificationsByStatusLock.Unlock()
		notificationsByStatus[notification.Status] = append(notificationsByStatus[notification.Status], &notification)
	}()
}

func sendMetrics(processingStartTime time.Time, notification models.Notification) {
	tags := map[string]string{
		"type":    notification.Type.Name,
		"subtype": notification.Subtype.String(),
		"status":  notification.Status.String(),
	}
	metrics.GlobalAppMetrics().GetOrCreateHistogram(
		processingMetricsPrefixName,
		tags,
		processingTimingMetricName,
		processingLatencyBuckets,
	).RecordDuration(time.Since(processingStartTime))
}

func (s *Service) processSingle(
	ctx context.Context,
	notification *models.Notification,
	notificationTx database.NotificationTransaction,
	iterationID uuid.UUID,
) {
	defer func() {
		if r := recover(); r != nil {
			s.logger.Error(
				"panic happened during processing single notification",
				log.Any("panic", r),
				log.UInt64("notificationID", notification.ID),
			)
			err := notificationTx.Commit(s.logger)
			s.logger.Error("failed to commit transaction after panic", log.Error(err), log.UInt64("notificationID", notification.ID))
		}
	}()
	s.logger.Info("start processing single notification")
	if !s.config.SendToUnsubscribed && !s.isRecipientSubscribed(notification, iterationID) {
		s.updateStatus(notification, notificationTx, models.NotificationStatusCancelled, iterationID)
		return
	}
	if processor, ok := s.processorByType[notification.Type.Name]; ok {
		s.logger.Info("got processor for notification")
		processor.Process(ctx, notification, notificationTx, iterationID)
		return
	}
	s.logger.Error(
		"failed to get processor by notification type",
		log.UInt64("notificationID", notification.ID),
		log.String("notificationType", notification.Type.Name),
	)
	s.updateStatus(notification, notificationTx, models.NotificationStatusFailed, iterationID)
}

func (s *Service) logResults(
	iterationID uuid.UUID,
	waitAllProcessed func(),
	notificationsByStatus map[models.NotificationStatus][]*models.Notification,
) {
	waitAllProcessed()
	for k, v := range notificationsByStatus {
		message := ""
		switch k {
		case models.NotificationStatusSent:
			message = "sent notifications"
		case models.NotificationStatusReadyToSend:
			message = "failed to send notifications"
		case models.NotificationStatusPostponed:
			message = "postponed notifications"
		case models.NotificationStatusPlanned:
			message = "unprocessed notifications"
		case models.NotificationStatusCancelled:
			message = "cancelled notifications"
		case models.NotificationStatusFailed:
			message = "completely failed notifications"
		}
		if message != "" {
			s.logger.Info(message, log.Int("count", len(v)), log.String("iterationID", iterationID.String()))
		}
	}
}

func (s *Service) isRecipientSubscribed(notification *models.Notification, iterationID uuid.UUID) bool {
	if !notification.Recipient.IsSubscribed {
		s.logger.Info(
			"notification should be cancelled due to its recipient isn't subscribed",
			log.UInt64("notificationID", notification.ID),
			log.String("notificationType", notification.Type.Name),
			log.String("iterationID", iterationID.String()),
		)
		return false
	}
	return true
}

func (s *Service) updateStatus(
	notification *models.Notification,
	notificationTx database.NotificationTransaction,
	newStatus models.NotificationStatus,
	iterationID uuid.UUID,
) {
	notification.Status = newStatus
	if notificationTx.Update(*notification, s.logger) != nil {
		s.logger.Error(
			"failed to set status %s for notification",
			log.Any("newStatus", newStatus),
			log.UInt64("notificationID", notification.ID),
			log.String("iterationID", iterationID.String()),
		)
	}
}
