package pullnotifications

import (
	"context"
	"encoding/json"
	"time"

	"github.com/jackc/pgtype"
	"github.com/jonboulle/clockwork"
	"google.golang.org/protobuf/types/known/timestamppb"

	pullnotifications "a.yandex-team.ru/travel/notifier/api/pull_notifications/v1"
	"a.yandex-team.ru/travel/notifier/internal/database"
	"a.yandex-team.ru/travel/notifier/internal/models"
	"a.yandex-team.ru/travel/notifier/internal/service/onlineregistration"
)

type Service struct {
	clock      clockwork.Clock
	repository *database.NotificationsRepository
}

func NewService(clock clockwork.Clock, repository *database.NotificationsRepository) *Service {
	return &Service{clock: clock, repository: repository}
}

func (s *Service) GetAll(ctx context.Context, passportID string) ([]*pullnotifications.PullNotification, error) {
	now := s.clock.Now().UTC()
	notifications, err := s.repository.GetPullNotificationsByPassportID(ctx, passportID, now)
	if err != nil {
		return nil, err
	}
	return s.mapNotifications(notifications), nil
}

func (s *Service) GetByOrderIDs(ctx context.Context, orderIDs []string) ([]*pullnotifications.PullNotification, error) {
	now := s.clock.Now().UTC()
	notifications, err := s.repository.GetPullNotificationsByOrderIDs(ctx, orderIDs, now)
	if err != nil {
		return nil, err
	}
	return s.mapNotifications(notifications), nil
}

func (s *Service) mapNotifications(rawNotifications []models.Notification) []*pullnotifications.PullNotification {
	notifications := make([]*pullnotifications.PullNotification, 0, len(rawNotifications))
	for i := 0; i < len(rawNotifications); i++ {
		notifications = append(notifications, &pullnotifications.PullNotification{
			Id:       int32(rawNotifications[i].ID),
			Type:     s.mapType(rawNotifications[i].Type),
			StartsAt: s.mapTimestamp(rawNotifications[i].NotifyAt),
			EndsAt:   s.mapTimestamp(rawNotifications[i].Deadline),
			Data:     s.mapPayload(rawNotifications[i].Type, rawNotifications[i].Payload),
			OrderId:  *rawNotifications[i].OrderID,
		})
	}
	return notifications
}

func (s *Service) mapType(notificationType models.NotificationType) pullnotifications.PullNotificationType {
	switch notificationType {
	case models.NotificationTypeOnlineRegistration:
		return pullnotifications.PullNotificationType_PULL_NOTIFICATION_TYPE_ONLINE_REGISTRATION
	default:
		return pullnotifications.PullNotificationType_PULL_NOTIFICATION_TYPE_UNKNOWN
	}
}

func (s *Service) mapTimestamp(time time.Time) *timestamppb.Timestamp {
	return timestamppb.New(time)
}

func (s *Service) mapPayload(notificationType models.NotificationType, payload *pgtype.JSON) *pullnotifications.NotificationData {
	switch notificationType {
	case models.NotificationTypeOnlineRegistration:
		model := &onlineregistration.Payload{}
		_ = json.Unmarshal(payload.Bytes, model)
		return &pullnotifications.NotificationData{
			Data: &pullnotifications.NotificationData_OnlineRegistration{
				OnlineRegistration: &pullnotifications.OnlineRegistration{
					MarketingFlightNumber: model.MarketingFlightNumber,
					OperationFlightNumber: model.OperatingFlightNumber,
					LocalDepartureTime:    model.LocalDepartureTime,
					DepartureStationId:    model.DepartureStationID,
					ArrivalStationId:      model.ArrivalStationID,
				},
			},
		}
	}
	return nil
}
