package queueobserver

import (
	"context"
	"strconv"
	"sync"
	"time"

	"github.com/jonboulle/clockwork"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/library/go/metrics"
	"a.yandex-team.ru/travel/notifier/internal/models"
	"a.yandex-team.ru/travel/notifier/internal/ytlock"
)

type NotificationsRepository interface {
	CountActualPlanned(context.Context, time.Time) (int64, error)
	CountPostponed(context.Context, time.Time) (int64, error)
	CountReadyToSend(context.Context) (int64, error)
	CountAllPlanned(ctx context.Context) (int64, error)
	CountAll(ctx context.Context) (int64, error)
}

type NotificationsQueueObserverService struct {
	logger                  log.Logger
	config                  Config
	clock                   clockwork.Clock
	notificationsRepository NotificationsRepository
	ytLock                  *ytlock.YtLock
	observeTicker           clockwork.Ticker
	lockAcquiringTicker     clockwork.Ticker
}

func NewNotificationsQueueObserverService(
	logger log.Logger,
	config Config,
	clock clockwork.Clock,
	notificationsRepository NotificationsRepository,
	ytLock *ytlock.YtLock,
) *NotificationsQueueObserverService {
	return &NotificationsQueueObserverService{
		logger:                  logger,
		config:                  config,
		clock:                   clock,
		notificationsRepository: notificationsRepository,
		ytLock:                  ytLock,
		observeTicker:           clock.NewTicker(config.ObserveInterval),
		lockAcquiringTicker:     clock.NewTicker(config.ObserveInterval),
	}
}

func (s *NotificationsQueueObserverService) Observe(ctx context.Context) {
	if s.config.Disabled {
		return
	}
	for range s.lockAcquiringTicker.Chan() {
		if s.tryAcquireLock(ctx) {
			break
		}
	}
	ctx, cancel := context.WithTimeout(context.Background(), s.config.YtLockTimeout)
	defer cancel()
	defer s.ytLock.Release(ctx)
	for range s.observeTicker.Chan() {
		s.observe()
	}
}

func (s *NotificationsQueueObserverService) Stop() {
	s.lockAcquiringTicker.Stop()
	s.observeTicker.Stop()
}

func (s *NotificationsQueueObserverService) observe() {
	actualPlannedTags := map[string]string{"status": models.NotificationStatusPlanned.String(), "only_actual": strconv.FormatBool(true)}
	allPlannedTags := map[string]string{"status": models.NotificationStatusPlanned.String(), "only_actual": strconv.FormatBool(false)}
	postponedTags := map[string]string{"status": models.NotificationStatusPostponed.String()}
	readyToSendTags := map[string]string{"status": models.NotificationStatusReadyToSend.String()}
	allNotificationsTags := map[string]string{"status": "all"}
	ctx, cancel := context.WithTimeout(context.Background(), s.config.ObserveTimeout)
	defer cancel()
	wg := sync.WaitGroup{}
	wg.Add(5)
	go func() {
		defer wg.Done()
		if count, err := s.notificationsRepository.CountActualPlanned(ctx, s.clock.Now()); err != nil {
			s.logger.Error("failed to count actual planned notifications", log.Error(err))
			s.sendFailuresMetric(actualPlannedTags)
		} else {
			s.sendNotificationsCountMetric(actualPlannedTags, count)
		}
	}()
	go func() {
		defer wg.Done()
		if count, err := s.notificationsRepository.CountAllPlanned(ctx); err != nil {
			s.logger.Error("failed to count all planned notifications", log.Error(err))
			s.sendFailuresMetric(allPlannedTags)
		} else {
			s.sendNotificationsCountMetric(allPlannedTags, count)
		}
	}()
	go func() {
		defer wg.Done()
		if count, err := s.notificationsRepository.CountPostponed(ctx, s.clock.Now()); err != nil {
			s.logger.Error("failed to count postponed notifications", log.Error(err))
			s.sendFailuresMetric(postponedTags)
		} else {
			s.sendNotificationsCountMetric(postponedTags, count)
		}
	}()
	go func() {
		defer wg.Done()
		if count, err := s.notificationsRepository.CountReadyToSend(ctx); err != nil {
			s.logger.Error("failed to count ready-to-send notifications", log.Error(err))
			s.sendFailuresMetric(readyToSendTags)
		} else {
			s.sendNotificationsCountMetric(readyToSendTags, count)
		}
	}()
	go func() {
		defer wg.Done()
		if count, err := s.notificationsRepository.CountAll(ctx); err != nil {
			s.logger.Error("failed to count all notifications", log.Error(err))
			s.sendFailuresMetric(allNotificationsTags)
		} else {
			s.sendNotificationsCountMetric(allNotificationsTags, count)
		}
	}()
	wg.Wait()
}

func (s *NotificationsQueueObserverService) tryAcquireLock(ctx context.Context) bool {
	err := s.ytLock.Acquire(ctx)
	return err == nil
}

func (s *NotificationsQueueObserverService) sendFailuresMetric(tags map[string]string) {
	metrics.GlobalAppMetrics().GetOrCreateCounter(queueObserverMetricsPrefixName, tags, queueObserverFailuresMetricName).Inc()
}

func (s *NotificationsQueueObserverService) sendNotificationsCountMetric(tags map[string]string, count int64) {
	metrics.GlobalAppMetrics().GetOrCreateGauge(
		queueObserverMetricsPrefixName,
		tags,
		queueObserverNotificationsCountMetricName,
	).Set(float64(count))
}
