package scheduler

import (
	"context"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/notifier/internal/models"
)

var (
	emptyRecipient           = models.Recipient{}
	emptyUser                = models.User{}
	emptyNotificationType    = models.NotificationType{}
	emptyNotificationChannel = models.NotificationChannel{}
	ErrNoRecipient           = xerrors.New("no recipient for notification")
	ErrNoUser                = xerrors.New("no user for notification")
	ErrNoNotificationType    = xerrors.New("no notification type")
	ErrNoNotificationChannel = xerrors.New("no notification channel")
	ErrNotifyAtInThePast     = xerrors.New("notifyAt in the past")
)

type NotificationsRepository interface {
	CancelPlannedForOrder(context.Context, string, models.NotificationType, models.DispatchType) ([]models.Notification, error)
	Create(context.Context, models.Notification) (*models.Notification, error)
	CreateMany(context.Context, []models.Notification) ([]models.Notification, error)
	AlreadySentForOrder(context.Context, string, models.NotificationType) ([]models.Notification, error)
}

type RecipientsRepository interface {
	Create(context.Context, models.Recipient) (*models.Recipient, error)
}

type Service struct {
	logger                  log.Logger
	notificationsRepository NotificationsRepository
	recipientsRepository    RecipientsRepository
}

func (s *Service) Schedule(ctx context.Context, notification models.Notification, now time.Time) (*models.Notification, error) {
	s.logger.Debugf("NotificationsRepository.Schedule called")
	if err := s.validateNotification(notification, now); err != nil {
		return nil, err
	}
	return s.notificationsRepository.Create(ctx, notification)
}

func (s *Service) ScheduleMany(ctx context.Context, notifications []models.Notification, now time.Time) ([]models.Notification, error) {
	s.logger.Debugf("NotificationsRepository.ScheduleMany called")
	for _, n := range notifications {
		if err := s.validateNotification(n, now); err != nil {
			return nil, err
		}
	}
	return s.notificationsRepository.CreateMany(ctx, notifications)
}

func (s *Service) validateNotification(notification models.Notification, now time.Time) error {
	if notification.DispatchType == models.DispatchTypePush && (notification.Recipient == nil || *notification.Recipient == emptyRecipient) {
		return ErrNoRecipient
	}
	if notification.DispatchType == models.DispatchTypePull && (notification.User == nil || *notification.User == emptyUser) {
		return ErrNoUser
	}
	if notification.Type == emptyNotificationType {
		return ErrNoNotificationType
	}
	if notification.Channel == emptyNotificationChannel {
		return ErrNoNotificationChannel
	}
	if notification.NotifyAt.Before(now) {
		return ErrNotifyAtInThePast
	}
	return nil
}

func (s *Service) CancelPlannedNotificationsByOrderID(ctx context.Context, orderID string, notificationType models.NotificationType, dispatchType models.DispatchType) ([]models.Notification, error) {
	notifications, err := s.notificationsRepository.CancelPlannedForOrder(ctx, orderID, notificationType, dispatchType)
	if err != nil {
		s.logger.Error("failed to cancel planned notifications for order", log.String("orderID", orderID), log.Error(err))
		return nil, err
	}
	if len(notifications) > 0 {
		notificationIDs := make([]uint64, 0, len(notifications))
		for _, n := range notifications {
			notificationIDs = append(notificationIDs, n.ID)
		}
		s.logger.Info(
			"cancelled planned notifications for order",
			log.String("orderID", orderID),
			log.UInt64s("notificationIDs", notificationIDs),
		)
	}
	return notifications, nil
}

func (s *Service) AlreadySentForOrder(ctx context.Context, orderID string) ([]models.Notification, error) {
	alreadySentNotifications, err := s.notificationsRepository.AlreadySentForOrder(ctx, orderID, models.NotificationTypePretrip)
	if err != nil {
		s.logger.Error(
			"failed to verify whether notifications have been sent already for order",
			log.Error(err),
			log.String("orderID", orderID),
		)
		return nil, err
	}
	return alreadySentNotifications, nil
}

func NewService(l log.Logger, notificationRepository NotificationsRepository, recipientsRepository RecipientsRepository) *Service {
	return &Service{logger: l, notificationsRepository: notificationRepository, recipientsRepository: recipientsRepository}
}
