package useractions

import (
	"crypto/sha256"
	"encoding/hex"
	"time"

	"a.yandex-team.ru/travel/library/go/unifiedagent"
	"a.yandex-team.ru/travel/notifier/internal/logging"
)

type ActionType string

const (
	logName = "subscriptions-user-action-log"

	ActionTypeSubscribe   ActionType = "subscribe"
	ActionTypeUnsubscribe ActionType = "unsubscribe"

	subscriptionTypePromo string = "promo"
)

type Logger struct {
	unifiedAgentLogger *logging.UnifiedAgentLogger
}

func NewLogger(unifiedAgentClient unifiedagent.Client) *Logger {
	return &Logger{unifiedAgentLogger: logging.NewUnifiedAgentLogger(logName, unifiedAgentClient)}
}

func (l *Logger) LogSubscribe(email, source, travelVertical, nationalVersion, language string, isPlusUser bool, experiments map[string]string, passportID, yandexUID string) error {
	record := &Record{
		Unixtime:         uint64(time.Now().Unix()),
		ActionType:       string(ActionTypeSubscribe),
		SubscriptionType: subscriptionTypePromo,
		Code:             "",
		NationalVersion:  nationalVersion,
		Language:         language,
		Email:            hashEmail(email),
		TravelVertical:   &travelVertical,
		Source:           &source,
		IsPlusUser:       isPlusUser,
		Experiments:      experiments,
		PassportID:       passportID,
		YandexUID:        yandexUID,
	}
	return l.unifiedAgentLogger.LogJSON(record)
}

func (l *Logger) LogUnsubscribe(email, nationalVersion, language string) error {
	record := &Record{
		Unixtime:         uint64(time.Now().Unix()),
		ActionType:       string(ActionTypeUnsubscribe),
		SubscriptionType: subscriptionTypePromo,
		Code:             "",
		NationalVersion:  nationalVersion,
		Language:         language,
		Email:            hashEmail(email),
		TravelVertical:   nil,
		Source:           nil,
	}
	return l.unifiedAgentLogger.LogJSON(record)
}

func hashEmail(email string) string {
	hasher := sha256.New()
	hasher.Write([]byte(email))
	return hex.EncodeToString(hasher.Sum(nil))
}
