package subscriptions

import (
	"context"
	"fmt"
	"strconv"
	"sync"
	"time"

	"github.com/gofrs/uuid"
	"golang.org/x/sync/errgroup"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/ctxlog"
	"a.yandex-team.ru/travel/library/go/metrics"
	"a.yandex-team.ru/travel/library/go/sender"
	"a.yandex-team.ru/travel/notifier/internal/database"
	"a.yandex-team.ru/travel/notifier/internal/models"
	"a.yandex-team.ru/travel/notifier/internal/orders"
	"a.yandex-team.ru/travel/notifier/internal/service/pretrip/interfaces"
	"a.yandex-team.ru/travel/notifier/internal/service/processing"
	ordersproto "a.yandex-team.ru/travel/orders/proto"
)

type Processor struct {
	logger                  log.Logger
	senderClient            interfaces.Sender
	ordersClient            *orders.Client
	notificationsRepository processing.NotificationsRepository
	recipientsRepository    interfaces.RecipientsRepository
	config                  Config
	txOptions               database.TransactionOptions
}

type processingContext struct {
	context.Context
	notification *models.Notification
	tx           database.NotificationTransaction
}

func NewProcessor(
	logger log.Logger,
	ordersClient *orders.Client,
	notificationsRepository processing.NotificationsRepository,
	recipientsRepository interfaces.RecipientsRepository,
	senderClient interfaces.Sender,
	config Config,
) *Processor {
	return &Processor{
		logger:                  logger.WithName("SubscriptionsProcessor"),
		ordersClient:            ordersClient,
		senderClient:            senderClient,
		notificationsRepository: notificationsRepository,
		recipientsRepository:    recipientsRepository,
		config:                  config,
		txOptions: database.TransactionOptions{
			StatementTimeout:                config.DBStatementTimeout,
			LockTimeout:                     config.DBLockTimeout,
			IdleInTransactionSessionTimeout: config.DBIdleInTransactionTimeout,
			WaitForLock:                     config.WaitForLockBeforeSending,
		},
	}
}

func (p *Processor) Process(
	ctx context.Context,
	notification *models.Notification,
	processingTx database.NotificationTransaction,
	_ uuid.UUID,
) {
	processingCtx, successful := p.prepareProcessingContext(ctx, notification, processingTx)
	if !successful {
		return
	}
	if p.updateNotificationStatus(processingCtx, models.NotificationStatusReadyToSend) != nil {
		return
	}

	if notification.PromoCode == nil {
		p.logger.Error("remind notification without promo code", ctxlog.ContextFields(processingCtx)...)
		return
	}

	if processingCtx.tx, successful = p.beginSendingTx(processingCtx); !successful {
		return
	}
	defer func() {
		if err := processingCtx.tx.Commit(p.logger); err != nil {
			p.logger.Error("failed to commit tx", append(ctxlog.ContextFields(ctx), log.Error(err))...)
		}
	}()

	promoCodeActivationAvailable, err := p.ordersClient.PromoCodeActivationAvailable(ctx, notification.PromoCode.Code)
	if err != nil {
		p.logger.Error(
			"failed to get promo code activation status for notification",
			append(ctxlog.ContextFields(ctx), log.Error(err))...,
		)
		return
	}

	mayBeActivated := false
	switch promoCodeActivationAvailable.Result {
	case ordersproto.EPromoCodeApplicationResultType_ART_SUCCESS:
		mayBeActivated = true
	}
	if !mayBeActivated {
		p.logger.Info("promo code may not be activated", ctxlog.ContextFields(processingCtx)...)
		p.cancelNotification(processingCtx)
		return
	}

	senderRequest := p.buildSenderRequest(notification, processingCtx)
	p.send(processingCtx, senderRequest)

	if err := processingTx.Commit(p.logger); err != nil {
		p.logger.Error(
			"failed to commit transaction after sending notification",
			append(ctxlog.ContextFields(processingCtx), log.Error(err))...,
		)
	}
}

func (p *Processor) beginSendingTransaction(
	ctx context.Context,
	waitGroup *sync.WaitGroup,
	notification *models.Notification,
	sendingTxChan chan<- database.NotificationTransaction,
) error {
	waitGroup.Done()
	sendingTx, err := p.notificationsRepository.BeginTransaction(ctx, *notification, p.txOptions)
	if err != nil {
		p.logger.Error("failed to begin sending transaction for notification", append(ctxlog.ContextFields(ctx), log.Error(err))...)
	}
	sendingTxChan <- sendingTx
	return err
}

func (p *Processor) beginSendingTx(processingCtx processingContext) (tx database.NotificationTransaction, successful bool) {
	errGroup := errgroup.Group{}
	sendingTxChan := make(chan database.NotificationTransaction, 1)
	waitGroup := sync.WaitGroup{}
	waitGroup.Add(1)
	errGroup.Go(
		func() error {
			return p.beginSendingTransaction(processingCtx, &waitGroup, processingCtx.notification, sendingTxChan)
		},
	)
	waitGroup.Wait()

	err := p.commitProcessingTx(processingCtx)
	if err != nil {
		return
	}
	err = errGroup.Wait()
	if err != nil {
		return
	}
	return <-sendingTxChan, true
}

func (p *Processor) buildSenderRequest(
	notification *models.Notification, processingCtx processingContext,
) sender.TransactionalRequest {
	return sender.TransactionalRequest{
		CampaignSlug: p.config.SenderCampaigns[processingCtx.notification.Subtype],
		ToEmail:      notification.Recipient.GetEmail(),
		SendAsync:    false,
		Args:         p.getPromoSendArgs(notification.PromoCode),
		Headers:      map[string]string{},
	}
}

func (p *Processor) cancelNotification(processingCtx processingContext) {
	err := p.updateNotificationStatus(processingCtx, models.NotificationStatusCancelled)
	if err != nil {
		p.logger.Error("failed to cancel notification", append(ctxlog.ContextFields(processingCtx), log.Error(err))...)
	}
}

func (p *Processor) commitProcessingTx(processingCtx processingContext) error {
	err := processingCtx.tx.Commit(p.logger)
	if err == nil {
		return nil
	}
	p.logger.Error("failed to commit before mailing", append(ctxlog.ContextFields(processingCtx), log.Error(err))...)
	tx, err := p.notificationsRepository.BeginTransaction(processingCtx, *processingCtx.notification, p.txOptions)
	if err != nil {
		p.logger.Error("failed to begin postpone transaction", append(ctxlog.ContextFields(processingCtx), log.Error(err))...)
		return err
	}
	p.postponeNotification(processingCtx, processingPhaseMetricName, postponementReasonFailedTxCommit)
	p.logger.Info("transaction is going to be committed", append(ctxlog.ContextFields(processingCtx), log.Error(err))...)
	if err := tx.Commit(p.logger); err != nil {
		p.logger.Error("failed to commit postpone transaction", append(ctxlog.ContextFields(processingCtx), log.Error(err))...)
		return err
	}
	return nil
}

func (p *Processor) getPromoSendArgs(promoCode *models.PromoCode) map[string]string {
	args := map[string]string{
		"code":           promoCode.Code,
		"nominal":        promoCode.Nominal,
		"min_total_cost": fmt.Sprintf("%d ₽", promoCode.MinTotalCost),
		"valid_till":     promoCode.ValidTill.Add(-24 * time.Hour).Format("2006-01-02"),
	}
	if promoCode.AddsUpWithOtherActions {
		args["adds_up_with_other_actions"] = "true"
	}
	return args
}

func (p *Processor) handleNotificationWithExpiredDeadline(processingCtx processingContext, phaseMetricName string) {
	metrics.GlobalAppMetrics().GetOrCreateCounter(
		subscriptionsMetricsPrefix,
		map[string]string{"phase": phaseMetricName, "subtype": processingCtx.notification.Subtype.String()},
		expiredNotificationMetricName,
	).Inc()
	p.logger.Info(
		"notification should be cancelled due to expired deadline",
		append(ctxlog.ContextFields(processingCtx), log.Time("notificationDeadline", processingCtx.notification.Deadline))...,
	)
	p.cancelNotification(processingCtx)
	if err := processingCtx.tx.Commit(p.logger); err != nil {
		p.logger.Error(
			"failed to cancel notification with expired deadline",
			append(ctxlog.ContextFields(processingCtx), log.Error(err))...,
		)
		return
	}
}

func (p *Processor) postponeNotification(processingCtx processingContext, phaseMetricName string, postponementReason string) {
	processingCtx.notification.NotifyAt = processingCtx.notification.NotifyAt.Add(p.config.PostponeInterval)
	if !processingCtx.notification.Deadline.IsZero() && processingCtx.notification.Deadline.Before(processingCtx.notification.NotifyAt) {
		p.handleNotificationWithExpiredDeadline(processingCtx, phaseMetricName)
		return
	}
	processingCtx.notification.Failures += 1
	processingCtx.notification.Status = models.NotificationStatusPostponed
	if err := processingCtx.tx.Update(*processingCtx.notification, p.logger); err != nil {
		p.logger.Error("failed to postpone notification", append(ctxlog.ContextFields(processingCtx), log.Error(err))...)
	} else {
		p.logger.Info("notification has been postponed", ctxlog.ContextFields(processingCtx)...)
		p.writePostponedNotificationsMetric(phaseMetricName, postponementReason, processingCtx.notification.Subtype)
	}
}

func (p *Processor) prepareProcessingContext(
	ctx context.Context,
	notification *models.Notification,
	processingTx database.NotificationTransaction,
) (processingCtx processingContext, successful bool) {
	ctx = ctxlog.WithFields(
		ctx,
		log.Int32("recipientID", *notification.RecipientID),
		log.UInt64("notificationID", notification.ID),
		log.String("notificationSubtype", notification.Subtype.String()),
	)
	processingCtx = processingContext{
		Context:      ctx,
		notification: notification,
		tx:           processingTx,
	}
	return processingCtx, true
}

func (p *Processor) requestSender(ctx context.Context, request sender.TransactionalRequest) {
	startSendingAt := time.Now()
	var err error
	defer func() {
		tags := map[string]string{"is_error": strconv.FormatBool(err != nil)}
		metrics.GlobalAppMetrics().GetOrCreateCounter(
			senderMetricsPrefix,
			tags,
			senderRequestsMetricName,
		).Inc()
		metrics.GlobalAppMetrics().GetOrCreateHistogram(
			senderMetricsPrefix,
			tags,
			senderTimingsMetricName,
			senderLatencyBuckets,
		).RecordDuration(time.Since(startSendingAt))
	}()

	if _, err = p.senderClient.SendTransactional(ctx, request); err != nil {
		p.logger.Error(
			"failed to send message",
			append(ctxlog.ContextFields(ctx), log.Error(err), log.Float64("duration_seconds", time.Since(startSendingAt).Seconds()))...,
		)
		return
	}
}

func (p *Processor) send(processingCtx processingContext, senderRequest sender.TransactionalRequest) {
	isSuccessful := false
	defer func() { p.writeSendingMetric(isSuccessful, processingCtx.notification.Subtype) }()
	p.requestSender(processingCtx, senderRequest)
	if p.updateNotificationStatus(processingCtx, models.NotificationStatusSent) != nil {
		return
	}
	isSuccessful = true
	p.logger.Info("notification has been sent successfully", ctxlog.ContextFields(processingCtx)...)
}

func (p *Processor) updateNotificationStatus(processingCtx processingContext, newStatus models.NotificationStatus) error {
	processingCtx.notification.Status = newStatus
	err := processingCtx.tx.Update(*processingCtx.notification, p.logger)
	if err != nil {
		p.logger.Error(
			"failed to update notification status",
			append(ctxlog.ContextFields(processingCtx), log.Error(err), log.Any("newStatus", newStatus))...,
		)
	} else {
		p.logger.Info(
			"notification status has been updated",
			append(ctxlog.ContextFields(processingCtx), log.Any("newStatus", newStatus))...,
		)
	}
	return err
}

func (p *Processor) writePostponedNotificationsMetric(phaseMetricName string, reason string, subtype models.NotificationSubtype) {
	metrics.GlobalAppMetrics().GetOrCreateCounter(
		subscriptionsMetricsPrefix,
		map[string]string{"phase": phaseMetricName, "reason": reason, "subtype": subtype.String()},
		postponedNotificationMetricName,
	).Inc()
}

func (p *Processor) writeSendingMetric(isSuccessful bool, subtype models.NotificationSubtype) {
	metricName := successfulSendingMetricName
	if !isSuccessful {
		metricName = failedSendingMetricName
	}
	metrics.GlobalAppMetrics().GetOrCreateCounter(
		subscriptionsMetricsPrefix,
		map[string]string{"phase": sendingPhaseMetricName, "subtype": subtype.String()},
		metricName,
	).Inc()
}
