package subscriptions

import (
	"context"
	"errors"
	"fmt"
	"hash/crc32"
	"math"
	"regexp"
	"strings"
	"sync"
	"time"

	"github.com/jonboulle/clockwork"
	"github.com/shopspring/decimal"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/hotels/proto/data_config/promo"
	"a.yandex-team.ru/travel/library/go/sender"
	pb "a.yandex-team.ru/travel/notifier/api/subscriptions/v1"
	"a.yandex-team.ru/travel/notifier/internal/models"
	"a.yandex-team.ru/travel/notifier/internal/service/subscriptions/logging/useractions"
	subscriptionmodels "a.yandex-team.ru/travel/notifier/internal/service/subscriptions/models"
	ordercommons "a.yandex-team.ru/travel/orders/proto"
	orderspromo "a.yandex-team.ru/travel/orders/proto/services/promo"
	travel_commons_proto "a.yandex-team.ru/travel/proto"
)

const promoForSubscriptionExp = "MARKETING_promo_for_subscription"
const promoCodePrefix = "HELLO-"

type RecipientsRepository interface {
	Get(context.Context, string) (*models.Recipient, error)
	GetOrCreateByEmail(context.Context, string) (*models.Recipient, error)
	Update(context.Context, models.Recipient) (*models.Recipient, error)
	GetByHash(context.Context, string) (*models.Recipient, error)
}

type PromoEventsRepository interface {
	GetAll() ([]*promo.TPromoEvent, error)
}

type PromoCampaignsRepository interface {
	CreatePromoCode(context.Context, string, string, time.Time, *orderspromo.TGetPromoActionResp) (*orderspromo.TCreatePromoCodeRsp, error)
	GetPromoActionDetails(context.Context, string) (response *orderspromo.TGetPromoActionResp, orderErr error)
	PromoCodeActivationAvailable(context.Context, string) (*orderspromo.TPromoCodeActivationAvailableResp, error)
}

type BetterPriceSubscriptionsRepository interface {
	Upsert(context.Context, models.BetterPriceSubscription) error
}

type UserActionsLogger interface {
	LogSubscribe(email, source, travelVertical, nationalVersion, language string, isPlusUser bool, experiments map[string]string, passportID, yandexUID string) error
	LogUnsubscribe(email, nationalVersion, language string) error
}

type UnsubscribeHashGenerator interface {
	Generate(string) string
}

type Scheduler interface {
	Schedule(context.Context, models.Notification, time.Time) (*models.Notification, error)
}

var senderEmailParamRegexp = regexp.MustCompile(`(.+email=).*?(["&].+)*$`)

type Service struct {
	config                             Config
	clock                              clockwork.Clock
	logger                             log.Logger
	recipientsRepository               RecipientsRepository
	betterPriceSubscriptionsRepository BetterPriceSubscriptionsRepository
	unsubscirbeHashGenerator           UnsubscribeHashGenerator
	senderClient                       sender.Client
	userActionsLogger                  UserActionsLogger
	promoEventsRepository              PromoEventsRepository
	promoCampaignsRepository           PromoCampaignsRepository
	notificationsScheduler             Scheduler
}

func NewService(
	logger log.Logger,
	recipientsRepository RecipientsRepository,
	betterPriceSubscriptionsRepository BetterPriceSubscriptionsRepository,
	unsubscirbeHashGenerator UnsubscribeHashGenerator,
	senderClient sender.Client,
	config Config,
	clock clockwork.Clock,
	userActionsLogger UserActionsLogger,
	promoEventsRepository PromoEventsRepository,
	promoCampaignsRepository PromoCampaignsRepository,
	notificationsScheduler Scheduler,
) *Service {
	return &Service{
		logger:                             logger.WithName("SubscriptionsService"),
		recipientsRepository:               recipientsRepository,
		unsubscirbeHashGenerator:           unsubscirbeHashGenerator,
		senderClient:                       senderClient,
		config:                             config,
		clock:                              clock,
		userActionsLogger:                  userActionsLogger,
		betterPriceSubscriptionsRepository: betterPriceSubscriptionsRepository,
		promoEventsRepository:              promoEventsRepository,
		promoCampaignsRepository:           promoCampaignsRepository,
		notificationsScheduler:             notificationsScheduler,
	}
}

func (s *Service) Subscribe(ctx context.Context, subscription Subscription) error {
	recipient, err := s.recipientsRepository.GetOrCreateByEmail(ctx, subscription.Email)
	if err != nil {
		s.logger.Error("failed to get or create recipient", log.Error(err))
		return err
	}

	if err = s.senderClient.Subscribe(ctx, s.buildSenderRequest(recipient.GetEmail())); err != nil {
		if errors.As(err, &sender.InvalidEmailError{}) {
			s.logger.Info("recipient won't be subscribed due to invalid email address", log.Int32("recipientID", recipient.ID))
			return nil
		}
		s.logger.Error(
			"failed to subscribe recipient in sender",
			log.Int32("recipientID", recipient.ID),
			log.String("error", removeEmail(err.Error())),
		)
		return err
	}

	err = s.execPromoActions(ctx, subscription, recipient)
	if err != nil {
		return err
	}

	recipient.Subscribe(
		subscription.Source,
		subscription.Vertical,
		subscription.NationalVersion,
		subscription.Language,
		subscription.Timezone,
		s.clock.Now(),
		s.unsubscirbeHashGenerator.Generate,
	)
	if _, err = s.recipientsRepository.Update(ctx, *recipient); err != nil {
		s.logger.Error("failed to subscribe recipient in database", log.Int32("recipientID", recipient.ID), log.Error(err))
		return err
	}
	err = s.userActionsLogger.LogSubscribe(
		recipient.GetEmail(),
		subscription.Source,
		subscription.Vertical,
		subscription.NationalVersion,
		subscription.Language,
		subscription.IsPlusUser,
		subscription.Experiments,
		subscription.PassportID,
		subscription.YandexUID,
	)
	if err != nil {
		s.logger.Error(
			"failed to write user-actions log",
			log.Error(err),
			log.Int32("recipientID", recipient.ID),
			log.Any("action", useractions.ActionTypeSubscribe),
		)
	}
	return nil
}

func (s *Service) Unsubscribe(ctx context.Context, hash string) error {
	recipient, err := s.recipientsRepository.GetByHash(ctx, hash)
	if err != nil {
		s.logger.Error("failed to get recipient by hash", log.String("hash", hash), log.Error(err))
		return err
	}
	if recipient == nil {
		s.logger.Info("no recipient has been found by hash", log.String("hash", hash), log.Error(err))
		return ErrNoRecipient
	}

	if err = s.senderClient.Unsubscribe(ctx, s.buildSenderRequest(recipient.GetEmail())); err != nil {
		s.logger.Error("failed to unsubscribe recipient in sender", log.Int32("recipientID", recipient.ID), log.Error(err))
		return err
	}

	recipient.Unsubscribe(s.clock.Now())
	if _, err = s.recipientsRepository.Update(ctx, *recipient); err != nil {
		s.logger.Error("failed to unsubscribe recipient", log.Int32("recipientID", recipient.ID), log.Error(err))
		return err
	}
	nationalVersion := ""
	if recipient.NationalVersion != nil {
		nationalVersion = *recipient.NationalVersion
	}
	language := ""
	if recipient.Language != nil {
		language = *recipient.Language
	}
	err = s.userActionsLogger.LogUnsubscribe(recipient.GetEmail(), nationalVersion, language)
	if err != nil {
		s.logger.Error(
			"failed to write user-actions log",
			log.Error(err),
			log.Int32("recipientID", recipient.ID),
			log.Any("action", useractions.ActionTypeUnsubscribe),
		)
	}
	return nil
}

func (s *Service) GetStatus(ctx context.Context, email string) (*models.SubscirptionStatus, error) {
	recipient, err := s.recipientsRepository.Get(ctx, email)
	if err != nil {
		s.logger.Error("failed to get recipient by email", log.Error(err))
		return nil, err
	}
	if recipient == nil {
		s.logger.Info("no recipient has been found by email")
		return &models.SubscirptionStatus{}, nil
	}
	subscriptionStatus := recipient.GetSubscriptionStatus()
	return &subscriptionStatus, nil
}

func (s *Service) GetPromoConfig(ctx context.Context, vertical string) (*pb.GetPromoConfigRsp, error) {
	promoEvents, err := s.promoEventsRepository.GetAll()
	if err != nil {
		s.logger.Error("failed to get promo events")
		return nil, err
	}
	for _, protoEvent := range promoEvents {
		event := models.PromoEventFromProto(protoEvent)
		if !event.Active(time.Now(), vertical) {
			continue
		}
		details, err := s.promoCampaignsRepository.GetPromoActionDetails(ctx, event.OrdersCampaignID)
		if err != nil {
			return nil, xerrors.Errorf("campaign=%s: %w", event.OrdersCampaignID, err)
		}
		if details == nil {
			return nil, xerrors.Errorf("campaign=%s: nil promo action details", event.OrdersCampaignID)
		}
		var generationConfig = details.GetGenerationConfig()
		if generationConfig == nil {
			return nil, xerrors.Errorf("nil generation config")
		}

		promoConfig := &pb.GetPromoConfigRsp{
			PromoCode: &pb.PromoCodeForSubscription{
				Type:   pb.EPromoCodeNominalType(generationConfig.GetNominalType()),
				Amount: generationConfig.GetNominal(),
			},
		}
		if minTotalCostTPrice := details.GetDiscountApplicationConfig().GetMinTotalCost(); minTotalCostTPrice != nil {
			minTotalCost, _ := decimal.New(minTotalCostTPrice.GetAmount(), -minTotalCostTPrice.GetPrecision()).Float64()
			promoConfig.PromoCode.MinTotalCost = minTotalCost
		}
		return promoConfig, nil
	}
	return &pb.GetPromoConfigRsp{}, nil
}

func (s *Service) SubscribeOnBetterPrice(ctx context.Context, subscription *subscriptionmodels.BetterPriceSubscription) error {
	recipient, err := s.recipientsRepository.GetOrCreateByEmail(ctx, subscription.Email)
	if err != nil {
		s.logger.Error("failed to get or create recipient", log.Error(err))
		return err
	}
	waitGroup := sync.WaitGroup{}
	waitGroup.Add(1)
	go func() {
		defer waitGroup.Done()
		err := s.Subscribe(ctx, Subscription{
			Email:           recipient.GetEmail(),
			Source:          subscription.Source,
			Vertical:        subscription.Vertical,
			Timezone:        subscription.UserTimezone,
			Language:        subscription.Language,
			NationalVersion: subscription.NationalVersion,
		})
		if err != nil {
			s.logger.Error("failed to subscribe for promo on better price subscription", log.Error(err))
		}
	}()
	subscriptionModel := s.mapBetterPriceSubscriptionToModel(subscription, recipient)
	err = s.betterPriceSubscriptionsRepository.Upsert(ctx, subscriptionModel)
	if err != nil {
		s.logger.Error("failed to upsert BetterPriceSubscriptions", log.Error(err))
		return err
	}
	waitGroup.Wait()
	return nil
}

func (s *Service) buildSenderRequest(email string) sender.UnsubscribeListRequest {
	return sender.UnsubscribeListRequest{
		Email:               email,
		UnsubscribeListSlug: s.config.SenderUnsubscribeListSlug,
	}
}

func (s *Service) buildSenderPromoRequest(
	email string,
	args map[string]string,
	campaignSlug string,
) sender.TransactionalRequest {
	return sender.TransactionalRequest{
		CampaignSlug: campaignSlug,
		ToEmail:      email,
		SendAsync:    false,
		Args:         args,
		Headers:      map[string]string{},
	}
}

func (s *Service) execPromoActions(ctx context.Context, subscription Subscription, recipient *models.Recipient) error {
	if subscription.Experiments[promoForSubscriptionExp] != "enabled" {
		s.logger.Info("experiment disabled")
		return nil
	}

	if recipient.IsSubscribed || recipient.SubscribedAt != nil || recipient.UnsubscribedAt != nil {
		s.logger.Info("recipient already was subscribed")
		return nil
	}

	matchedPromoEvents, err := s.getMatchedPromoEvents(subscription)
	if err != nil {
		return err
	}
	s.logger.Debug("got matched promo events", log.Any("matchedPromoEvents", matchedPromoEvents))

	now := s.clock.Now()
	for _, event := range matchedPromoEvents {
		s.logger.Debug("processing promo event", log.Any("event", event))

		validTill := now.Add(s.config.PromoCodeValidDuration)
		validTill = time.Date(
			validTill.Year(), validTill.Month(), validTill.Day(), 0, 0, 0, 0, validTill.Location(),
		).Add(time.Hour * 24)
		notifyAt := validTill.Add(-s.config.PromoCodeNotifyStart)
		deadline := notifyAt.Add(-s.config.PromoCodeNotifyEnd)

		promoCode, err := s.generatePromoCode(ctx, subscription, event, validTill)
		if err != nil {
			s.logger.Error("failed to generate promo", log.Error(err))
			return err
		}
		s.logger.Debug("promo code generated", log.Any("promoCode", promoCode))

		campaignSlug := s.config.SenderCampaigns[models.NotificationPromoSend]
		senderRequest := s.buildSenderPromoRequest(recipient.GetEmail(), s.getPromoSendArgs(promoCode), campaignSlug)

		s.logger.Debug("sending promo code", log.Any("senderRequest", senderRequest))
		if _, err = s.senderClient.SendTransactional(ctx, senderRequest); err != nil {
			s.logger.Error("failed to send promo code", log.Error(err))
			return err
		}
		s.logger.Debug("promo code successfully sent")

		if !s.config.SendPromoNotifications {
			s.logger.Debug("skip promo remind notification scheduling")
			continue
		}

		notification, err := s.preparePromoRemindNotification(recipient, promoCode, notifyAt, deadline)
		if err != nil {
			s.logger.Error("failed to prepare promo remind notification", log.Error(err))
			return err
		}
		s.logger.Debug("promo remind notification prepared", log.Any("notification", notification))

		_, err = s.notificationsScheduler.Schedule(ctx, *notification, now)
		if err != nil {
			s.logger.Error("failed to schedule promo remind notification", log.Error(err))
			return err
		}
		s.logger.Debug("promo remind notification scheduled", log.Any("notification", notification))
	}
	return nil
}

func (s *Service) getCode(id string) string {
	return fmt.Sprintf("%s%08X", promoCodePrefix, crc32.ChecksumIEEE([]byte(id)))
}

func (s *Service) getMatchedPromoEvents(subscription Subscription) ([]models.PromoEvent, error) {
	promoEventsProto, err := s.promoEventsRepository.GetAll()
	if err != nil {
		s.logger.Error("failed to get promo events")
		return nil, err
	}
	var promoEvents []models.PromoEvent
	for _, protoEvent := range promoEventsProto {
		event := models.PromoEventFromProto(protoEvent)
		if !event.Active(time.Now(), subscription.Vertical) {
			continue
		}
		for _, eventSource := range event.Sources {
			if subscription.Source == eventSource {
				promoEvents = append(promoEvents, event)
				break
			}
		}
	}
	return promoEvents, nil
}

func (s Service) getNominal(nominalType ordercommons.EPromoCodeNominalType, nominalValue int) (string, error) {
	var nominal string
	switch nominalType {
	case ordercommons.EPromoCodeNominalType_NT_VALUE:
		nominal = fmt.Sprintf("%d ₽", nominalValue)
	case ordercommons.EPromoCodeNominalType_NT_PERCENT:
		nominal = fmt.Sprintf("%d%%", nominalValue)
	default:
		return "", xerrors.Errorf("unknown nominal type %+v", nominalType)
	}
	return nominal, nil
}

func (s *Service) getPriceValueRounded(price *travel_commons_proto.TPrice) (int, error) {
	if price == nil {
		return 0, xerrors.Errorf("price is nil")
	}
	if price.GetCurrency() != travel_commons_proto.ECurrency_C_RUB {
		return 0, xerrors.Errorf("price currency == %v, expected ECurrency_C_RUB", price.GetCurrency())
	}
	return int(float64(price.GetAmount()) / math.Pow(10, float64(price.GetPrecision()))), nil
}

func (s *Service) getPromoSendArgs(promoCode models.PromoCode) map[string]string {
	args := map[string]string{
		"code":           promoCode.Code,
		"nominal":        promoCode.Nominal,
		"min_total_cost": fmt.Sprintf("%d ₽", promoCode.MinTotalCost),
		"valid_till":     promoCode.ValidTill.Add(-24 * time.Hour).Format("2006-01-02"),
	}
	if promoCode.AddsUpWithOtherActions {
		args["adds_up_with_other_actions"] = "true"
	}
	return args
}

func (s *Service) generatePromoCode(
	ctx context.Context,
	subscription Subscription,
	event models.PromoEvent,
	validTill time.Time,
) (models.PromoCode, error) {
	userID := subscription.YandexUID
	if userID == "" {
		userID = subscription.PassportID
	}
	if userID == "" {
		return models.PromoCode{}, xerrors.Errorf(
			"campaign=%s: subscription has neither YandexUID nor PassportID", event.OrdersCampaignID,
		)
	}
	code := s.getCode(userID)
	promoActionDetails, err := s.promoCampaignsRepository.GetPromoActionDetails(ctx, event.OrdersCampaignID)
	if err != nil {
		return models.PromoCode{}, err
	}
	if promoActionDetails == nil {
		return models.PromoCode{}, xerrors.Errorf(
			"campaign=%s: nil promo action details", event.OrdersCampaignID,
		)
	}
	generationConfig := promoActionDetails.GetGenerationConfig()
	if generationConfig == nil {
		return models.PromoCode{}, xerrors.Errorf(
			"campaign=%s: nil generation config", event.OrdersCampaignID,
		)
	}
	discountApplicationConfig := promoActionDetails.GetDiscountApplicationConfig()
	if discountApplicationConfig == nil {
		return models.PromoCode{}, xerrors.Errorf(
			"campaign=%s: nil discount application config", event.OrdersCampaignID,
		)
	}
	minTotalCost, err := s.getPriceValueRounded(discountApplicationConfig.GetMinTotalCost())
	if err != nil {
		return models.PromoCode{}, xerrors.Errorf(
			"campaign=%s: failed to get min total cost, %v", event.OrdersCampaignID, err,
		)
	}
	nominal, err := s.getNominal(generationConfig.GetNominalType(), int(generationConfig.GetNominal()))
	if err != nil {
		return models.PromoCode{}, xerrors.Errorf(
			"campaign=%s: failed to get nominal, %v", event.OrdersCampaignID, err,
		)
	}

	_, err = s.promoCampaignsRepository.CreatePromoCode(
		ctx, event.OrdersCampaignID, code, validTill, promoActionDetails,
	)
	if err != nil {
		return models.PromoCode{}, err
	}

	promoCode := models.PromoCode{
		Code:                   code,
		Nominal:                nominal,
		AddsUpWithOtherActions: discountApplicationConfig.GetAddsUpWithOtherActions(),
		MinTotalCost:           minTotalCost,
		ValidTill:              validTill,
	}
	return promoCode, nil
}

func (s *Service) mapBetterPriceSubscriptionToModel(
	subscription *subscriptionmodels.BetterPriceSubscription,
	recipient *models.Recipient,
) models.BetterPriceSubscription {
	return models.BetterPriceSubscription{
		Recipient: *recipient,
		Variant: models.Variant{
			ForwardKey:  stringifyFlights(subscription.Variant.ForwardFlights),
			BackwardKey: stringifyFlights(subscription.Variant.BackwardFlights),
		},
		PriceWithoutBaggageValue:             subscription.TariffWithoutBaggage.Price.Value,
		PriceWithoutBaggageCurrency:          subscription.TariffWithoutBaggage.Price.Currency,
		ExchangedPriceWithoutBaggageValue:    subscription.TariffWithoutBaggage.ExchangedPrice.Value,
		ExchangedPriceWithoutBaggageCurrency: subscription.TariffWithoutBaggage.ExchangedPrice.Currency,
		PriceWithBaggageValue:                subscription.TariffWithBaggage.Price.Value,
		PriceWithBaggageCurrency:             subscription.TariffWithBaggage.Price.Currency,
		ExchangedPriceWithBaggageValue:       subscription.TariffWithBaggage.ExchangedPrice.Value,
		ExchangedPriceWithBaggageCurrency:    subscription.TariffWithBaggage.ExchangedPrice.Currency,
		FromPointKey:                         subscription.FromPointKey,
		ToPointKey:                           subscription.ToPointKey,
		NationalVersion:                      subscription.NationalVersion,
		Language:                             subscription.Language,
		DateForward:                          subscription.DateForward,
		DateBackward:                         subscription.DateBackward,
		Adults:                               subscription.Passengers.Adults,
		Children:                             subscription.Passengers.Children,
		Infants:                              subscription.Passengers.Infants,
		ServiceClass:                         subscription.ServiceClass.String(),
		WithBaggage:                          subscription.WithBaggage,
	}
}

func (s *Service) preparePromoRemindNotification(
	recipient *models.Recipient,
	promoCode models.PromoCode,
	notifyAt time.Time,
	deadline time.Time,
) (*models.Notification, error) {
	notification := models.NewNotification(
		notifyAt,
		deadline,
		models.NotificationStatusPlanned,
		models.NotificationTypePromoCode,
		models.NotificationChannelEmail,
		models.DispatchTypePush,
	).WithSubtype(models.NotificationPromoRemind).WithPromoCode(promoCode).WithRecipient(*recipient)
	return &notification, nil
}

var ErrNoRecipient = fmt.Errorf("recipient not found")

func removeEmail(errorMessage string) string {
	return senderEmailParamRegexp.ReplaceAllString(errorMessage, "${1}HIDDEN_EMAIL${2}")
}

func stringifyFlights(flights []subscriptionmodels.Flight) string {
	parts := make([]string, 0, len(flights))
	for _, f := range flights {
		parts = append(parts, fmt.Sprintf("%s_%s_%d_%d_%s", f.CompanyCode, f.Number, f.FromStationID, f.ToStationID, f.LocalDepartureDatetime))
	}
	return strings.Join(parts, ";")
}
