package structs

import (
	"encoding/json"
	"time"
)

type (
	HotelPayload struct {
		RegionGeoID   int               `json:"region_geo_id"`
		RequestParams RequestParams     `json:"request_params"`
		HotelsList    []HotelsBlockItem `json:"hotels_list"`
	}

	HotelsBlockItem struct {
		Hotel      Hotel       `json:"hotel"`
		MinPrice   *HotelPrice `json:"min_price"`
		LandingURL string      `json:"landing_url"`
		Badges     []Badge     `json:"badges"`
	}

	Badge struct {
		ID   string `json:"id"`
		Text string `json:"text"`
	}

	Hotel struct {
		Permalink            string        `json:"permalink"`
		Name                 string        `json:"name"`
		Category             HotelCategory `json:"category"`
		Address              string        `json:"address"`
		Rating               float32       `json:"rating"`
		Stars                int           `json:"stars"`
		Images               []HotelImage  `json:"images"`
		MainAmenities        []MainAmenity `json:"main_amenities"`
		TotalTextReviewCount int           `json:"total_text_review_count"`
	}

	MainAmenity struct {
		ID   string `json:"id"`
		Name string `json:"name"`
	}

	HotelPrice struct {
		Value    float64 `json:"value"`
		Currency string  `json:"currency"`
	}

	HotelCategory struct {
		ID   string `json:"id"`
		Name string `json:"name"`
	}

	HotelImage struct {
		URLTemplate string            `json:"url_template"`
		ID          string            `json:"id"`
		Sizes       []HotelImageSizes `json:"sizes"`
		Tags        []string          `json:"tags"`
	}

	HotelImageSizes struct {
		Size   string `json:"size"`
		Width  int    `json:"width"`
		Height int    `json:"height"`
	}

	RequestParams struct {
		CheckinDate  Date  `json:"checkin_date"`
		CheckoutDate Date  `json:"checkout_date"`
		Nights       int   `json:"nights"`
		Adults       int   `json:"adults"`
		ChildrenAges []int `json:"children_ages"`
	}
)

type Date time.Time

func (d *Date) UnmarshalJSON(b []byte) (err error) {
	var strTime string
	err = json.Unmarshal(b, &strTime)
	if err != nil {
		return err
	}
	parsed, err := time.Parse("2006-01-02", strTime)
	if err != nil {
		return err
	}
	*d = Date(parsed)
	return nil
}

func NewDate(year, month, day int) Date {
	return Date(time.Date(year, time.Month(month), day, 0, 0, 0, 0, time.UTC))
}
