package handler

import (
	"context"
	"fmt"

	"github.com/opentracing/opentracing-go"
	"google.golang.org/grpc"
	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/status"

	"a.yandex-team.ru/library/go/core/log"
	pb "a.yandex-team.ru/travel/notifier/api/subscriptions/v1"
	"a.yandex-team.ru/travel/notifier/internal/models"
	"a.yandex-team.ru/travel/notifier/internal/service/subscriptions"
	subscriptionmodels "a.yandex-team.ru/travel/notifier/internal/service/subscriptions/models"
)

type SubscriptionsService interface {
	GetStatus(ctx context.Context, email string) (*models.SubscirptionStatus, error)
	GetPromoConfig(ctx context.Context, vertical string) (*pb.GetPromoConfigRsp, error)
	Subscribe(ctx context.Context, subscription subscriptions.Subscription) error
	Unsubscribe(ctx context.Context, hash string) error
	SubscribeOnBetterPrice(ctx context.Context, subscription *subscriptionmodels.BetterPriceSubscription) error
}

type BetterPriceSubscriptionMapper interface {
	Map(*pb.BetterPriceSubscribeReq) *subscriptionmodels.BetterPriceSubscription
}

type GRPCSubcriptionsHandler struct {
	logger                        log.Logger
	subscriptionsService          SubscriptionsService
	betterPriceSubscriptionMapper BetterPriceSubscriptionMapper
}

func NewGRPCSubscriptionsHandler(
	logger log.Logger,
	subscriber SubscriptionsService,
	betterPriceSubscriptionMapper BetterPriceSubscriptionMapper,
) *GRPCSubcriptionsHandler {
	return &GRPCSubcriptionsHandler{
		logger:                        logger.WithName("GRPCSubcriptionsHandler"),
		subscriptionsService:          subscriber,
		betterPriceSubscriptionMapper: betterPriceSubscriptionMapper,
	}
}

func (h *GRPCSubcriptionsHandler) Subscribe(ctx context.Context, request *pb.SubscribeReq) (
	response *pb.SubscribeRsp,
	err error,
) {
	span, _ := opentracing.StartSpanFromContext(ctx, "internal.handler.GRPCSubcriptionsHandler:Subscribe")
	defer span.Finish()

	subscription := subscriptions.Subscription{
		Email:           request.Email,
		Source:          request.Source,
		Vertical:        request.Vertical,
		Timezone:        request.Timezone,
		Language:        request.Language,
		NationalVersion: request.NationalVersion,
		YandexUID:       request.YandexUid,
		PassportID:      request.PassportId,
		IsPlusUser:      request.IsPlusUser,
		Experiments:     request.GetExperiments(),
	}
	if err := h.subscriptionsService.Subscribe(ctx, subscription); err != nil {
		return nil, status.Error(codes.FailedPrecondition, err.Error())
	}
	return &pb.SubscribeRsp{}, nil
}

func (h *GRPCSubcriptionsHandler) Unsubscribe(ctx context.Context, request *pb.UnsubscribeReq) (
	response *pb.UnsubscribeRsp,
	err error,
) {
	span, _ := opentracing.StartSpanFromContext(ctx, "internal.handler.GRPCSubcriptionsHandler:Unsubscribe")
	defer span.Finish()

	if err := h.subscriptionsService.Unsubscribe(ctx, request.Hash); err != nil {
		if err == subscriptions.ErrNoRecipient {
			return nil, status.Error(codes.NotFound, fmt.Sprintf("no recipients for hash %s", request.Hash))
		}
		return nil, status.Error(codes.FailedPrecondition, err.Error())
	}
	return &pb.UnsubscribeRsp{}, nil
}

func (h *GRPCSubcriptionsHandler) GetStatus(ctx context.Context, request *pb.GetStatusReq) (*pb.GetStatusRsp, error) {
	span, _ := opentracing.StartSpanFromContext(ctx, "internal.handler.GRPCSubscriptionsHandler:GetStatus")
	defer span.Finish()

	subscriptionStatus, err := h.subscriptionsService.GetStatus(ctx, request.Email)
	if err != nil {
		if err == subscriptions.ErrNoRecipient {
			return nil, status.Error(codes.NotFound, fmt.Sprintf("no recipients for email %s", request.Email))
		}
		return nil, status.Error(codes.FailedPrecondition, err.Error())
	}
	return &pb.GetStatusRsp{
		IsSubscribed:   subscriptionStatus.IsSubscribed,
		Source:         subscriptionStatus.Source,
		Vertical:       subscriptionStatus.Vertical,
		SubscribedAt:   subscriptionStatus.SubscribedAt,
		UnsubscribedAt: subscriptionStatus.UnsubscribedAt,
	}, nil
}

func (h *GRPCSubcriptionsHandler) GetPromoConfig(
	ctx context.Context,
	request *pb.GetPromoConfigReq,
) (*pb.GetPromoConfigRsp, error) {
	span, _ := opentracing.StartSpanFromContext(ctx, "internal.handler.GRPCSubscriptionsHandler::GetPromoConfig")
	defer span.Finish()

	response, err := h.subscriptionsService.GetPromoConfig(ctx, request.GetVertical())
	if err != nil {
		h.logger.Error("GetPromoConfig", log.Error(err))
		return nil, status.Error(codes.Internal, err.Error())
	}
	return response, nil
}

func (h *GRPCSubcriptionsHandler) SubscribeOnBetterPrice(
	ctx context.Context,
	request *pb.BetterPriceSubscribeReq,
) (*pb.BetterPriceSubscribeRsp, error) {
	span, _ := opentracing.StartSpanFromContext(ctx, "internal.handler.GRPCSubcriptionsHandler:SubscribeOnBetterPrice")
	defer span.Finish()
	if err := h.subscriptionsService.SubscribeOnBetterPrice(ctx, h.betterPriceSubscriptionMapper.Map(request)); err != nil {
		return nil, status.Error(codes.FailedPrecondition, err.Error())
	}
	return &pb.BetterPriceSubscribeRsp{}, nil
}

func (h *GRPCSubcriptionsHandler) GetServiceRegisterer() func(*grpc.Server) {
	return func(server *grpc.Server) {
		pb.RegisterSubscriptionsServiceServer(server, h)
	}
}
