package base

import (
	"context"
	"sync"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/library/go/errutil"
	"a.yandex-team.ru/travel/notifier/internal/ytdicts/ytdictconfig"
	"a.yandex-team.ru/yt/go/ypath"
	"a.yandex-team.ru/yt/go/yt"
)

type LastModifiedTableLoader struct {
	serviceConfig ytdictconfig.ServiceConfig
	logger        log.Logger
	aliveProxies  AliveProxiesProvider
}

func NewLastModifiedTableLoader(config ytdictconfig.ServiceConfig, logger log.Logger, aliveProxies AliveProxiesProvider) *LastModifiedTableLoader {
	return &LastModifiedTableLoader{
		serviceConfig: config,
		logger:        logger,
		aliveProxies:  aliveProxies,
	}
}

func (l *LastModifiedTableLoader) Populate(ctx context.Context, adder RowAdder) (err error) {
	defer errutil.Wrap(&err, "Populate")
	l.logger.Debug("Populating", log.Any("path", l.serviceConfig.TablePath))
	ctx, cancel := context.WithTimeout(ctx, l.serviceConfig.GetReadTimeout())
	defer cancel()

	clients, err := l.aliveProxies.AliveProxyClients(ctx)
	if err != nil {
		return errutil.NewTracedError(xerrors.Errorf("can not get alive proxies: %w", err))
	}
	proxy, err := l.proxyWithLastModifiedTable(ctx, clients, l.serviceConfig.TablePath)
	if err != nil {
		return err
	}
	return ReadTable(ctx, clients[proxy], l.serviceConfig.TablePath, adder)
}

func (l *LastModifiedTableLoader) proxyWithLastModifiedTable(ctx context.Context, clients map[string]yt.Client, path ypath.Path) (proxy string, err error) {
	defer errutil.Wrap(&err, "proxyWithLastModifiedTable(path=%s)", path)
	type proxyAndModificationTime struct {
		proxy            string
		modificationTime time.Time
	}
	var results = make(chan proxyAndModificationTime, len(clients))
	wg := sync.WaitGroup{}
	wg.Add(len(clients))
	for proxy, client := range clients {
		go func(proxy string, client yt.Client) {
			defer wg.Done()
			var modificationTime time.Time
			err = client.GetNode(ctx, path.Attr("modification_time"), &modificationTime, nil)
			if err != nil {

				return
			}
			results <- proxyAndModificationTime{proxy, modificationTime}
		}(proxy, client)
	}
	wg.Wait()
	close(results)
	var maxResult *proxyAndModificationTime
	for result := range results {
		if maxResult == nil || maxResult.modificationTime.Before(result.modificationTime) {
			maxResult = &result
		}
	}
	if maxResult == nil {
		if err != nil {
			return "", errutil.NewTracedError(xerrors.Errorf("could not find any proxy with latest mod time: %w", err))
		} else {
			return "", errutil.NewTracedError(xerrors.Errorf("could not find any proxy with latest mod time: unknown error"))
		}
	}
	return maxResult.proxy, nil
}
