package base

import (
	"context"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/yt/go/ypath"
	"a.yandex-team.ru/yt/go/yt"
	"a.yandex-team.ru/yt/go/yt/ythttp"
)

const defaultPingTimeout = 10 * time.Second

type AliveProxiesProvider interface {
	AliveProxyClients(ctx context.Context) (map[string]yt.Client, error)
}

type aliveProxiesProvider struct {
	Config          yt.Config
	Proxies         []string
	_pingTimeout    time.Duration
	logger          log.Logger
	ytClientBuilder func(c *yt.Config) (yt.Client, error)
}

func NewAliveProxiesProvider(config yt.Config, proxies []string, logger log.Logger) *aliveProxiesProvider {
	p := &aliveProxiesProvider{
		Config:          config,
		Proxies:         proxies,
		logger:          logger,
		ytClientBuilder: ythttp.NewClient,
	}
	return p
}

func (p *aliveProxiesProvider) AliveProxyClients(ctx context.Context) (map[string]yt.Client, error) {
	var lastErr error
	proxies := make(map[string]yt.Client, len(p.Proxies))
	for _, proxy := range p.Proxies {
		client, err := p.proxyClient(proxy)
		if err != nil {
			lastErr = err
			continue
		}
		err = p.ping(ctx, client)
		if err != nil {
			lastErr = err
			continue
		}
		proxies[proxy] = client
	}
	if len(proxies) == 0 {
		return nil, lastErr
	}
	return proxies, nil
}

func (p *aliveProxiesProvider) pingTimeout() time.Duration {
	if p._pingTimeout > 0 {
		return p._pingTimeout
	}
	return defaultPingTimeout
}

func (p *aliveProxiesProvider) proxyClient(proxy string) (yt.Client, error) {
	configCopy := p.Config
	configCopy.Proxy = proxy
	return p.ytClientBuilder(&configCopy)
}

func (p *aliveProxiesProvider) ping(ctx context.Context, client yt.Client) error {
	ctx, cancel := context.WithTimeout(ctx, p.pingTimeout())
	defer cancel()

	if client == nil {
		return xerrors.Errorf("client is nil")
	}
	homeExists, err := client.NodeExists(ctx, ypath.Path("//home"), nil)
	if err != nil {
		return err
	}
	if !homeExists {
		return xerrors.Errorf("node //home does not exist on cluster")
	}
	return nil
}
