package ytdicts

import (
	"context"
	"fmt"
	"strings"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/library/go/errutil"
	"a.yandex-team.ru/travel/notifier/internal/ytdicts/base"
	"a.yandex-team.ru/travel/notifier/internal/ytdicts/ytdictconfig"
	"a.yandex-team.ru/yt/go/yt"
)

type Config struct {
	IgnoreStartupErrors bool                       `yaml:"ignore_startup_errors"`
	PromoEvents         ytdictconfig.ServiceConfig `yaml:"promo_events"`
}

type UpdatableRepository interface {
	AddRow(scanner base.YtTableRowScanner) error
	UpdateFromSource(ctx context.Context, updater base.RepositoryUpdater) error
	Ready() bool
}

type Registry struct {
	config Config
	logger log.Logger

	promoEvents           *PromoEventsRepository
	updatableRepositories map[UpdatableRepository]ytdictconfig.ServiceConfig
}

func (r *Registry) GetPromoEventsRepository() *PromoEventsRepository {
	return r.promoEvents
}

func NewRegistry(c Config, logger log.Logger) (r *Registry) {
	promoEventsRepository := NewPromoEventsRepository()
	r = &Registry{
		promoEvents: promoEventsRepository,
		config:      c,
		logger:      logger,
		updatableRepositories: map[UpdatableRepository]ytdictconfig.ServiceConfig{
			promoEventsRepository: c.PromoEvents,
		},
	}
	return r
}

func (r *Registry) BackgroundRun() {
	for repository, config := range r.updatableRepositories {
		if !config.Enabled {
			continue
		}
		go func(repository UpdatableRepository, config ytdictconfig.ServiceConfig) {
			_ = r.runRepositoryUpdater(context.Background(), repository, config, false)
		}(repository, config)
	}
}

func (r *Registry) UpdateRegistryOnce(ctx context.Context) (err error) {
	defer errutil.Wrap(&err, "UpdateRegistryOnce")
	r.logger.Info("Running single registry update for all repositories")
	var errorsText []string
	for repository, config := range r.updatableRepositories {
		if !config.Enabled {
			r.logger.Warnf("Repository %T is disabled", repository)
			continue
		}
		r.logger.Infof("Updating %T repository", repository)
		if err := r.runRepositoryUpdater(ctx, repository, config, true); err != nil {
			if r.config.IgnoreStartupErrors {
				errorsText = append(errorsText, fmt.Sprintf("%T: %v", repository, err))
				continue
			}
			return err
		}
		r.logger.Infof("Update %T success", repository)
	}
	if len(errorsText) > 0 {
		return xerrors.Errorf("errors updating registry: %s", strings.Join(errorsText, ","))
	}
	return nil
}

func (r *Registry) runRepositoryUpdater(ctx context.Context, repository UpdatableRepository, config ytdictconfig.ServiceConfig, once bool) error {
	ticker := time.NewTicker(config.GetUpdateInterval())
	defer ticker.Stop()
	aliveProxies := base.NewAliveProxiesProvider(
		yt.Config{
			Token:             config.Token,
			ReadTokenFromFile: true,
		},
		config.Proxies,
		r.logger.WithName("AliveProxiesProvider"),
	)
	tableLoader := base.NewLastModifiedTableLoader(
		config,
		r.logger.WithName("LastModifiedTableLoader"),
		aliveProxies,
	)
	if once {
		return repository.UpdateFromSource(ctx, tableLoader)
	}
	for range ticker.C {
		err := repository.UpdateFromSource(ctx, tableLoader)
		if err != nil {
			log.Error(err)
		}
	}
	return nil
}

func (r *Registry) AllReady() bool {
	for repository := range r.updatableRepositories {
		if !repository.Ready() {
			return false
		}
	}
	return true
}
