import {Agent} from 'https';
import axios from 'axios';

import logRequest from './helpers/logRequest';
import handleError from './helpers/handleError';
import getCommonHeaders from './helpers/getCommonHeaders';

const moment = require('moment');

interface IDateRange {
    created_at_from_filter?: string;
    created_at_to_filter?: string;
}

function getDateRangeProps(fromDate: any, toDate: any): IDateRange {
    const dateRange: IDateRange = {};

    if (fromDate) {
        dateRange.created_at_from_filter = moment(Number(fromDate))
            .utc()
            .format();
    }

    if (toDate) {
        dateRange.created_at_to_filter = moment(Number(toDate)).utc().format();
    }

    return dateRange;
}

export default async function (req: any, res: any): Promise<void> {
    const {
        page = '0',
        count = '30',
        partnerId = '',
        prettyId = '',
        email = '',
        phone = '',
        name = '',
        card = '',
        token = '',
        status = 'OS_UNKNOWN',
        displayType = 'DT_UNKNOWN',
        partner = 'PT_UNKNOWN',
        fromDate = '',
        toDate = '',
        isBroken = '',
        paymentScheduleType = 'PST_UNKNOWN',
        sortType = 'SD_DESC',
        ticketNumber = '',
        yandexUid = '',
        carrier = '',
        referralPartnerId = '',
    } = req.query;

    if (
        !/^\d+$/.test(page) ||
        !/^\d+$/.test(count) ||
        !/^[a-zA-Z0-9-_]*$/.test(partnerId) ||
        !/^[a-zA-Z0-9-_]*$/.test(prettyId) ||
        !/^[a-zA-Z0-9-_@.]*$/.test(email) ||
        !/^[0-9+]*$/.test(phone) ||
        !/^[а-яА-Яa-zA-Z0-9-_.,\s]*$/.test(name) ||
        !/^[0-9*]*$/.test(card) ||
        !/^[A-Z_]*$/.test(status) ||
        !/^[A-Z_]*$/.test(displayType) ||
        !/^[A-Z_]*$/.test(partner) ||
        !/^\d*$/.test(fromDate) ||
        !/^\d*$/.test(toDate) ||
        !/^[A-Z_]*$/.test(paymentScheduleType) ||
        !/^[a-zA-Z0-9-_]*$/.test(ticketNumber) ||
        !/^[a-zA-Z0-9-_]*$/.test(yandexUid) ||
        !/^[a-zA-Z0-9-_]*$/.test(carrier) ||
        !/^[a-zA-Z0-9-_]*$/.test(referralPartnerId) ||
        (isBroken && !/^(true|false)$/.test(String(isBroken))) ||
        !/^[A-Z_]*$/.test(sortType)
    ) {
        return handleError(req, res, new Error('Invalid params.'));
    }

    const {ordersApi} = req.utils.config;

    try {
        const {
            data: {orders, quantity, filter_values},
        } = await axios.post(
            `${ordersApi}/api/admin/v1/list_orders`,
            {
                limit: count,
                offset: page * count,
                provider_id_filter: partnerId,
                pretty_id_filter: prettyId,
                email_filter: email,
                phone_filter: phone,
                passenger_names_filter: name,
                order_state_filter: status,
                display_order_type_filter: displayType,
                partner_filter: partner,
                card_mask_filter: card,
                is_broken_filter: isBroken,
                payment_schedule_type_filter: paymentScheduleType,
                purchase_token_filter: token,
                ticket_number_filter: ticketNumber,
                yandex_uid_filter: yandexUid,
                carrier_filter: carrier,
                referral_partner_id_filter: referralPartnerId,
                ...getDateRangeProps(fromDate, toDate),
                sorters: [
                    {
                        sort_direction: sortType,
                        sort_field: 'SF_CREATED_AT',
                    },
                ],
                snippets: ['S_PRIVATE_INFO'],
            },
            {
                headers: getCommonHeaders(req),
                httpsAgent: new Agent({
                    rejectUnauthorized: false,
                }),
            },
        );

        res.send({
            orders,
            quantity,
            order_state_values: [
                'OS_UNKNOWN',
                ...filter_values.order_state_values,
            ],
            display_order_type_values: [
                'DT_UNKNOWN',
                ...filter_values.display_order_type_values,
            ],
            order_partner_values: [
                'PT_UNKNOWN',
                ...filter_values.partner_values,
            ],
        });

        logRequest(req, res);
    } catch (err) {
        handleError(req, res, err);
    }
}
