'use strict';

import axios from 'axios';
import {Agent} from 'https';

import logRequest from './helpers/logRequest';
import handleError from './helpers/handleError';
import getCommonHeaders from './helpers/getCommonHeaders';

export default async function (req: any, res: any): Promise<void> {
    const {id: orderId} = req.params;

    const {refundType, reason, adminActionToken, refundsByFiscalItem} =
        req.body;

    if (!/^[a-z0-9-]*$/.test(orderId)) {
        return handleError(req, res, new Error('Invalid params.'));
    }

    const {ordersApi} = req.utils.config;

    try {
        const config = {
            headers: getCommonHeaders(req),
            httpsAgent: new Agent({
                rejectUnauthorized: false,
            }),
        };

        let data;

        if (refundType === 'ALL') {
            data = await axios.post(
                `${ordersApi}/api/admin/v1/manual_full_money_refund`,
                {
                    admin_action_token: adminActionToken,
                    order_id: orderId,
                    reason,
                },
                config,
            );
        } else if (refundType === 'FEE') {
            data = await axios.post(
                `${ordersApi}/api/admin/v1/manual_train_fee_refund`,
                {
                    admin_action_token: adminActionToken,
                    order_id: orderId,
                    reason,
                },
                config,
            );
        } else {
            data = await axios.post(
                `${ordersApi}/api/admin/v1/manual_money_refund`,
                {
                    admin_action_token: adminActionToken,
                    order_id: orderId,
                    reason,
                    refunds_by_fiscal_item: refundsByFiscalItem,
                },
                config,
            );
        }

        const {data: dataData} = data;

        res.send(dataData);

        logRequest(req, res);
    } catch (err) {
        handleError(req, res, err);
    }
}
