import React, {useCallback, useRef, useState} from 'react';
import {block} from 'bem-cn';

import Heading from 'components/basic/Heading/Heading';
import Modal from 'components/basic/Modal/Modal';
import ActionButton from 'components/lego/ActionButton/ActionButton';
import Textinput from 'components/lego/Textinput/Textinput';
import Tooltip from 'components/lego/Tooltip/Tooltip';

import './index.scss';

const b = block('ConfirmModal');

interface IOwnProps {
    className?: string;
    isVisible: boolean;
    onConfirm: (reason: string) => void;
    children?: React.ReactNode;
    requestState: {
        isFetched: boolean;
        hasError: boolean;
        isLoading: boolean;
    };
    onClose: () => void;
    title: string;
    successText: string;
    errorText: string;
    withReason?: boolean;
    isInvalidReason?: boolean;
}

const ConfirmModal: React.FC<IOwnProps> = props => {
    const {
        isVisible,
        onConfirm,
        children,
        requestState,
        onClose,
        successText,
        title,
        errorText,
        withReason,
        isInvalidReason = false,
    } = props;

    const [isNew, setNew] = useState(true);
    const [reason, setReason] = useState('');

    const handleClose = useCallback(() => {
        onClose();
        setNew(false);
        setReason('');
    }, [onClose]);
    const handleConfirm = useCallback(() => {
        onConfirm(reason);
        setNew(false);
    }, [onConfirm, reason]);
    const handleChangeReason = useCallback(
        event => setReason(event.target.value),
        [setReason],
    );

    const reasonInputRef = useRef(null);
    const buttonText = requestState.hasError ? 'Я настаиваю' : 'Подтвердить';
    const contentByRequestStatus =
        requestState.isFetched && !requestState.hasError && !isNew ? (
            <p className={b('Success')}>{successText}</p>
        ) : (
            <ActionButton
                onClick={handleConfirm}
                disabled={requestState.isLoading}
                progress={requestState.isLoading}
            >
                {buttonText}
            </ActionButton>
        );

    return (
        <Modal
            isVisible={isVisible}
            containerClassName={b('Content')}
            onClose={handleClose}
        >
            <Heading level="4">{title}</Heading>

            {children}

            {withReason && (
                <>
                    <div className={b('ReasonInput')}>
                        Причина с ссылкой на тикет
                        <Textinput
                            innerRef={reasonInputRef}
                            value={reason}
                            onChange={handleChangeReason}
                            placeholder="Ссылка на тикет"
                        />
                    </div>
                    <Tooltip
                        anchor={reasonInputRef}
                        visible={isInvalidReason}
                        state="alert"
                        hasTail
                        direction="bottom"
                    >
                        Некорректная причина
                    </Tooltip>
                </>
            )}

            {requestState.hasError && (
                <div className={b('Error')}>{errorText}</div>
            )}

            {contentByRequestStatus}
        </Modal>
    );
};

export default ConfirmModal;
