import React, {useCallback, useRef, useState} from 'react';
import {useDispatch, useSelector} from 'react-redux';
import {useToasts} from 'react-toast-notifications';
import {block} from 'bem-cn';
import {modifyHotelOrderDetailsRequest} from 'redux/reducers/order/actions';

import {IStore} from 'redux/reducers/types';

import getHumanFormatWithoutTime from 'lib/time/getHumanFormatWithoutTime';
import getRequestFormatWithoutTime from 'lib/time/getRequestFormatWithoutTime';
import getDatesFromRange from 'lib/time/getDatesFromRange';

import EditIcon from 'components/basic/icons/EditIcon';
import DefaultButton from 'components/lego/DefaultButton/DefaultButton';
import Tooltip from 'components/lego/Tooltip/Tooltip';
import ConfirmModal from 'components/Order/ConfirmModal/ConfirmModal';
import DatePicker from 'components/Order/HotelDatesOfStates/DatePicker/DatePicker';

import './index.scss';

const b = block('HotelDatesOfState');

interface IOwnProps {
    orderId: string;
    adminActionToken: string;
    checkinDate: string;
    checkoutDate: string;
    canModifyHotelOrderDetails: boolean;
}

const HotelDatesOfState: React.FC<IOwnProps> = props => {
    const {
        checkinDate: originCheckinDate,
        checkoutDate: originCheckoutDate,
        orderId,
        adminActionToken,
        canModifyHotelOrderDetails,
    } = props;

    const [isModalVisible, setModalVisible] = useState(false);
    const [checkinDate, setCheckinDate] = useState<Date>(
        new Date(originCheckinDate),
    );
    const [checkoutDate, setCheckoutDate] = useState<Date>(
        new Date(originCheckoutDate),
    );
    const [isInvalidDates, setInvalidDates] = useState(false);
    const [isInvalidReason, setInvalidReason] = useState(false);

    const {isLoading, isFetched, error} = useSelector(
        (state: IStore) => state.order.modifyHotelOrderDetails,
    );

    const dispatch = useDispatch();
    const {addToast} = useToasts();

    const handleConfirm = useCallback(
        (reason: string) => {
            const isInvalidCurrentDates =
                checkinDate.getTime() >= checkoutDate.getTime();

            setInvalidDates(isInvalidCurrentDates);

            const isInvalidCurrentReason = !reason;

            setInvalidReason(isInvalidCurrentReason);

            if (isInvalidCurrentDates || isInvalidCurrentReason) {
                return;
            }

            dispatch(
                modifyHotelOrderDetailsRequest({
                    orderId,
                    adminActionToken,
                    dates: {
                        checkin_date: getRequestFormatWithoutTime(checkinDate),
                        checkout_date:
                            getRequestFormatWithoutTime(checkoutDate),
                    },
                    addToast,
                    reason,
                }),
            );
        },
        [
            orderId,
            adminActionToken,
            checkinDate,
            checkoutDate,
            addToast,
            dispatch,
        ],
    );

    const daysOfStateContent =
        checkinDate && checkoutDate
            ? `Количество ночей: ${getDatesFromRange(
                  checkinDate,
                  checkoutDate,
              )}`
            : 'Невалидные даты';

    const checkinDatePickerRef = useRef(null);
    const checkoutDatePickerRef = useRef(null);

    return (
        <div className={b()}>
            <div className={b('DateContainer')}>
                <div>
                    Дата заезда: {getHumanFormatWithoutTime(originCheckinDate)}
                </div>
                <div>
                    Дата выезда: {getHumanFormatWithoutTime(originCheckoutDate)}
                </div>
            </div>
            {canModifyHotelOrderDetails && (
                <>
                    <div className={b('ButtonContainer')}>
                        <DefaultButton
                            onClick={() => setModalVisible(true)}
                            icon={EditIcon}
                        />
                    </div>
                    <ConfirmModal
                        isVisible={isModalVisible}
                        onConfirm={handleConfirm}
                        onClose={() => setModalVisible(false)}
                        requestState={{
                            isFetched: isFetched,
                            isLoading: isLoading,
                            hasError: Boolean(error),
                        }}
                        title="Изменить даты проживания"
                        successText="Данные успешно обновлены"
                        errorText="Не удалось обновить данные"
                        withReason={true}
                        isInvalidReason={isInvalidReason}
                    >
                        <div className={b('Controls')}>
                            <div className={b('DatePickerWrapper')}>
                                <div className={b('DatePickerWrapperName')}>
                                    Дата заезда
                                </div>
                                <DatePicker
                                    innerRef={checkinDatePickerRef}
                                    format="dd.MM.y"
                                    openCalendarOnFocus={false}
                                    isOpen={false}
                                    required
                                    value={checkinDate}
                                    onChange={setCheckinDate}
                                />
                                <Tooltip
                                    anchor={checkinDatePickerRef}
                                    visible={isInvalidDates}
                                    state="alert"
                                    hasTail
                                >
                                    Некорректная дата
                                </Tooltip>
                            </div>
                            <div className={b('DatePickerWrapper')}>
                                <div className={b('DatePickerWrapperName')}>
                                    Дата выезда
                                </div>
                                <DatePicker
                                    innerRef={checkoutDatePickerRef}
                                    format="dd.MM.y"
                                    openCalendarOnFocus={false}
                                    isOpen={false}
                                    required
                                    value={checkoutDate}
                                    onChange={setCheckoutDate}
                                    className={b('DatePicker')}
                                />
                                <Tooltip
                                    anchor={checkoutDatePickerRef}
                                    visible={isInvalidDates}
                                    state="alert"
                                    hasTail
                                >
                                    Некорректная дата
                                </Tooltip>
                            </div>
                        </div>
                        <div>
                            Проживание с{' '}
                            {getHumanFormatWithoutTime(checkinDate)} по{' '}
                            {getHumanFormatWithoutTime(checkoutDate)}
                        </div>
                        <div className={b('Days')}>{daysOfStateContent}</div>
                    </ConfirmModal>
                </>
            )}
        </div>
    );
};

export default HotelDatesOfState;
