import React, {useCallback, useState} from 'react';
import {block} from 'bem-cn';

import {IGuest} from 'components/Order/HotelGuestsInfo/types';

import CloseOutlineIcon from 'components/basic/icons/CloseOutlineIcon';
import Textinput from 'components/lego/Textinput/Textinput';
import Tumbler from 'components/lego/Tumbler/Tumbler';

import './index.scss';

const b = block('HotelGuest');

interface IOwnProps {
    index: number;
    guest: IGuest;
    onChange: (guest: IGuest, index: number) => void;
    onDelete: (index: number) => void;
    withDelete: boolean;
}

const HotelGuest: React.FC<IOwnProps> = props => {
    const {
        index,
        guest: {
            id,
            first_name: firstName,
            last_name: lastName,
            age,
            child: isChild,
        },
        onChange,
        onDelete,
        withDelete,
    } = props;

    const [currentFirstName, setCurrentFirstName] = useState(firstName);
    const [currentLastName, setCurrentLastName] = useState(lastName);
    const [currentAge, setCurrentAge] = useState<string>(
        age ? String(age) : '',
    );
    const [isCurrentChild, setCurrentChild] = useState(Boolean(isChild));

    const notifyParent = useCallback(
        (
            guestFirstName: string,
            guestLastName: string,
            guestIsChild: boolean,
            guestAge?: string,
        ) => {
            onChange(
                {
                    id,
                    first_name: guestFirstName,
                    last_name: guestLastName,
                    age: guestAge === undefined ? undefined : Number(guestAge),
                    child: guestIsChild,
                },
                index,
            );
        },
        [onChange, index, id],
    );

    const handleFirstNameChange = useCallback(
        event => {
            setCurrentFirstName(event.target.value);
            notifyParent(
                event.target.value,
                currentLastName,
                isCurrentChild,
                currentAge,
            );
        },
        [currentLastName, currentAge, isCurrentChild, notifyParent],
    );

    const handleLastNameChange = useCallback(
        event => {
            setCurrentLastName(event.target.value);
            notifyParent(
                currentFirstName,
                event.target.value,
                isCurrentChild,
                currentAge,
            );
        },
        [currentFirstName, currentAge, isCurrentChild, notifyParent],
    );

    const handleChildChange = useCallback(() => {
        setCurrentAge('');
        setCurrentChild(!isCurrentChild);

        notifyParent(currentFirstName, currentLastName, !isCurrentChild, '');
    }, [currentFirstName, currentLastName, isCurrentChild, notifyParent]);

    const handleAgeChange = useCallback(
        event => {
            setCurrentAge(event.target.value);
            notifyParent(
                currentFirstName,
                currentLastName,
                isCurrentChild,
                event.target.value,
            );
        },
        [currentFirstName, currentLastName, isCurrentChild, notifyParent],
    );

    const handleDelete = useCallback(() => onDelete(index), [index, onDelete]);

    return (
        <div className={b()}>
            <span className={b('Label')}>Имя</span>
            <span className={b('FirstNameInput')}>
                <Textinput
                    value={currentFirstName}
                    onChange={handleFirstNameChange}
                />
            </span>
            <span className={b('Label')}>Фамилия</span>
            <span className={b('LastNameInput')}>
                <Textinput
                    value={currentLastName}
                    onChange={handleLastNameChange}
                />
            </span>
            <span className={b('Label')}>Ребенок</span>
            <span className={b('IsChildTumbler')}>
                <Tumbler
                    checked={isCurrentChild}
                    onChange={handleChildChange}
                />
            </span>
            <span className={b('Label')}>Возраст</span>
            <span className={b('AgeInput')}>
                <Textinput
                    disabled={!isCurrentChild}
                    value={currentAge ? currentAge : ''}
                    onChange={handleAgeChange}
                />
            </span>
            {withDelete && (
                <span className={b('DeleteButton')} onClick={handleDelete}>
                    <CloseOutlineIcon />
                </span>
            )}
        </div>
    );
};

export default HotelGuest;
