import React, {useCallback, useEffect, useState} from 'react';
import {useDispatch, useSelector} from 'react-redux';
import {useToasts} from 'react-toast-notifications';
import {block} from 'bem-cn';
import {
    modifyHotelOrderDetailsRequest,
    orderRequest,
} from 'redux/reducers/order/actions';

import {IStore} from 'redux/reducers/types';
import {IHotelOrderGuest} from 'redux/reducers/types/hotel';
import {IGuest} from 'components/Order/HotelGuestsInfo/types';

import validateGuests from 'lib/guest/validateGuests';
import checkUpdates from 'lib/guest/checkUpdates';

import HotelGuest from 'components/Order/HotelGuestsInfo/HotelGuest/HotelGuest';
import AddButtonIcon from 'components/basic/icons/AddButtonIcon';
import ConfirmModal from 'components/Order/ConfirmModal/ConfirmModal';
import DefaultButton from 'components/lego/DefaultButton/DefaultButton';
import ActionButton from 'components/lego/ActionButton/ActionButton';

import './index.scss';

const b = block('HotelGuests');

interface IOwnProps {
    orderId: string;
    adminActionToken: string;
    guests: IHotelOrderGuest[];
}

const HotelGuests: React.FC<IOwnProps> = props => {
    const {orderId, adminActionToken} = props;
    const originGuests = props.guests.map(guest => ({
        id: new Date().getTime(),
        ...guest,
    }));

    const {isLoading, isFetched, error} = useSelector(
        (state: IStore) => state.order.modifyHotelOrderDetails,
    );

    const dispatch = useDispatch();
    const {addToast} = useToasts();

    const [guests, setGuests] = useState<IGuest[]>(originGuests);
    const [isVisibleModal, setVisibleModal] = useState(false);
    const [isNeedRefresh, setNeedRefresh] = useState(false);
    const [isValidGuests, setValidGuests] = useState(true);
    const [hasChanges, setHasChanges] = useState(false);

    const updateValidGuests = useCallback(
        updatedGuests => setValidGuests(validateGuests(updatedGuests)),
        [],
    );
    const updateHasChanges = useCallback(
        updatedGuests =>
            setHasChanges(checkUpdates(originGuests, updatedGuests)),
        [originGuests],
    );

    useEffect(() => {
        if (!isNeedRefresh) {
            return;
        }

        setTimeout(
            () =>
                dispatch(
                    orderRequest({orderId, needToFetchPersonalData: false}),
                ),
            2000,
        );
        setNeedRefresh(false);
    }, [dispatch, orderId, isNeedRefresh]);

    const onGuestChange = useCallback(
        (guest: IGuest, index) => {
            guests[index] = guest;
            setGuests(guests);
            updateValidGuests(guests);
            updateHasChanges(guests);
        },
        [guests, updateValidGuests, updateHasChanges],
    );

    const handleConfirm = useCallback(
        (reason: string) => {
            dispatch(
                modifyHotelOrderDetailsRequest({
                    orderId,
                    adminActionToken,
                    guests: {guests},
                    addToast,
                    reason,
                }),
            );
            setNeedRefresh(true);
        },
        [orderId, adminActionToken, guests, addToast, dispatch],
    );

    const handleOnDelete = useCallback(
        (index: number) => {
            const resultGuests =
                index + 1 <= guests.length
                    ? [...guests.slice(0, index), ...guests.slice(index + 1)]
                    : [...guests.slice(0, index)];

            setGuests(resultGuests);
            updateValidGuests(resultGuests);
            updateHasChanges(resultGuests);
        },
        [guests, updateValidGuests, updateHasChanges],
    );

    const handleReset = useCallback(() => {
        setGuests([...originGuests]);
        setValidGuests(true);
        setHasChanges(false);
    }, [originGuests]);

    const handleAddGuest = useCallback(() => {
        const resultGuests = [
            ...guests,
            {
                first_name: '',
                last_name: '',
                child: false,
                id: new Date().getTime(),
            },
        ];

        setGuests(resultGuests);
        setHasChanges(true);
        updateValidGuests(resultGuests);
    }, [guests, updateValidGuests]);

    return (
        <div className={b()}>
            {guests.map((guest, index) => (
                <HotelGuest
                    key={guest.id}
                    index={index}
                    guest={guest}
                    onChange={onGuestChange}
                    onDelete={handleOnDelete}
                    withDelete={guests.length > 1}
                />
            ))}

            <div className={b('AddButton')}>
                <DefaultButton onClick={handleAddGuest} icon={AddButtonIcon} />
            </div>

            <div className={b('ActionButtons')}>
                <ActionButton
                    onClick={() => setVisibleModal(true)}
                    disabled={!hasChanges || !isValidGuests}
                >
                    Сохранить
                </ActionButton>
                <DefaultButton onClick={handleReset} disabled={!hasChanges}>
                    Сбросить изменения
                </DefaultButton>
            </div>
            <ConfirmModal
                isVisible={isVisibleModal}
                onConfirm={handleConfirm}
                onClose={() => setVisibleModal(false)}
                requestState={{isLoading, isFetched, hasError: Boolean(error)}}
                title="Обновить список гостей?"
                successText="Данные успешно сохранены"
                errorText="Не удалось обновить данные"
                withReason={true}
            />
        </div>
    );
};

export default HotelGuests;
