import React, {useCallback, useState} from 'react';
import {useSelector} from 'react-redux';
import {block} from 'bem-cn';
import {noop} from 'lodash';

import {ICalculatedHotelMoneyOnlyRefund, IStore} from 'redux/reducers/types';

import {usePartialRefundAmount} from 'components/Order/helpers/usePartialRefundAmount';

import MoneyInput from 'components/basic/MoneyInput/MoneyInput';
import ActionButton from 'components/lego/ActionButton/ActionButton';
import RadioButton from 'components/lego/RadioButton/RadioButton';
import Textarea from 'components/lego/Textarea/Textarea';

import './index.scss';

const b = block('HotelMoneyOnlyRefundDeveloperForm');

interface IOwnProps {
    className?: string;
    calculatedRefund: ICalculatedHotelMoneyOnlyRefund;
    onAction: (
        requestedMoneyValue: number,
        requestedFinEvent: boolean,
        requestedRefundUserMoney: boolean,
        reason: string,
        requestedConfirmText: React.ReactNode,
    ) => void;
    isLoading: boolean;
}

enum EGenerateFinEventsRadioButtonValues {
    YES = 'Yes',
    NO = 'No',
}

enum ERefundUserMoneyRadioButtonValues {
    YES = 'Yes',
    NO = 'No',
}

enum EMoneyRefundTypes {
    FULL = 'full',
    AMOUNT = 'amount',
}

const HotelMoneyOnlyRefundForm: React.FC<IOwnProps> = props => {
    const {className, calculatedRefund, onAction} = props;

    const {isLoading} = useSelector(
        (state: IStore) => state.order.refundHotelMoneyOnly,
    );

    const [isGenerateFinEvent, setGenerateFinEvent] = useState<boolean | null>(
        null,
    );
    const [isRefundUserMoney, setRefundUserMoney] = useState<boolean | null>(
        null,
    );
    const [moneyRefundAmountValue, setMoneyRefundAmountValue] = useState(1);
    const [moneyRefundType, setMoneyRefundType] = useState(
        EMoneyRefundTypes.FULL,
    );
    const [isMoneyRefundAmountValueValid, setMoneyRefundAmountValueValid] =
        useState(false);
    const [reason, setReason] = useState('');

    const {partialRefundAmount, hasDiscount} = usePartialRefundAmount({
        moneyRefundType,
        inputValue: moneyRefundAmountValue,
    });

    const refundMoney =
        moneyRefundType === EMoneyRefundTypes.AMOUNT
            ? partialRefundAmount
            : calculatedRefund.remaining_amount.value;

    const handleGenerateFinEventChange = useCallback(event => {
        setGenerateFinEvent(
            event.target.value === EGenerateFinEventsRadioButtonValues.YES,
        );
    }, []);
    const handleRefundUserMoneyChange = useCallback(event => {
        setRefundUserMoney(
            event.target.value === EGenerateFinEventsRadioButtonValues.YES,
        );
    }, []);
    const handleMoneyRefundAmountValueChange = useCallback(
        (isValid: boolean, value: number) => {
            if (isValid) {
                setMoneyRefundAmountValue(value);
            }

            setMoneyRefundAmountValueValid(isValid);
        },
        [],
    );
    const handleMoneyRefundTypeChange = useCallback(event => {
        setMoneyRefundType(event.target.value);
        setMoneyRefundAmountValueValid(true);
        setMoneyRefundAmountValue(1);
    }, []);

    const isRefundOptionsValid =
        isGenerateFinEvent !== null &&
        isRefundUserMoney !== null &&
        (isGenerateFinEvent || isRefundUserMoney) &&
        (moneyRefundType !== EMoneyRefundTypes.AMOUNT ||
            isMoneyRefundAmountValueValid);

    const generateFinEventRadioButtonValue = isGenerateFinEvent
        ? EGenerateFinEventsRadioButtonValues.YES
        : EGenerateFinEventsRadioButtonValues.NO;

    const refundUserMoneyRadioButtonValue = isRefundUserMoney
        ? ERefundUserMoneyRadioButtonValues.YES
        : ERefundUserMoneyRadioButtonValues.NO;

    const moneyRefundTypeOptions = [
        {value: EMoneyRefundTypes.FULL, children: 'Полностью'},
        {value: EMoneyRefundTypes.AMOUNT, children: 'Частично'},
    ];

    const handleClick = useCallback(() => {
        onAction(
            refundMoney,
            Boolean(isGenerateFinEvent),
            Boolean(isRefundUserMoney),
            reason,
            <React.Fragment>
                <p>
                    Будут возвращены{' '}
                    <span className={b('Bold')}>{refundMoney} руб </span>
                    из{' '}
                    <span className={b('Bold')}>
                        {calculatedRefund.remaining_amount.value} руб{' '}
                    </span>
                    оставшихся, при общей сумме{' '}
                    <span className={b('Bold')}>
                        {calculatedRefund.total_amount.value} руб.{' '}
                    </span>
                    После возврата останется{' '}
                    <span className={b('Bold')}>
                        {calculatedRefund.remaining_amount.value - refundMoney}{' '}
                        руб.
                    </span>
                </p>
                <p>
                    Заказ <span className={b('Bold')}>не</span> будет отменён
                </p>
                <p>
                    Пользователю{' '}
                    {isRefundUserMoney ? null : (
                        <span className={b('Bold')}>не </span>
                    )}
                    будут возвращены деньги.
                </p>
                <p>
                    Финансовые события{' '}
                    {isGenerateFinEvent ? null : (
                        <span className={b('Bold')}>не </span>
                    )}
                    будут сгенерированы
                </p>
            </React.Fragment>,
        );
    }, [
        calculatedRefund,
        onAction,
        refundMoney,
        isGenerateFinEvent,
        reason,
        isRefundUserMoney,
    ]);

    const handleReasonChange = useCallback(
        event => setReason(event.target.value),
        [],
    );

    return (
        <div className={b.mix(className)}>
            <div className={b('Option')}>
                Вернуть деньги отельера?
                <RadioButton
                    value={
                        isGenerateFinEvent === null
                            ? ''
                            : generateFinEventRadioButtonValue
                    }
                    options={[
                        {
                            value: EGenerateFinEventsRadioButtonValues.YES,
                            children: 'Да',
                        },
                        {
                            value: EGenerateFinEventsRadioButtonValues.NO,
                            children: 'Нет',
                        },
                    ]}
                    onChange={handleGenerateFinEventChange}
                />
            </div>

            <div className={b('Option')}>
                Вернуть деньги пользователю?
                <RadioButton
                    value={
                        isRefundUserMoney === null
                            ? ''
                            : refundUserMoneyRadioButtonValue
                    }
                    options={[
                        {
                            value: ERefundUserMoneyRadioButtonValues.YES,
                            children: 'Да',
                        },
                        {
                            value: ERefundUserMoneyRadioButtonValues.NO,
                            children: 'Нет',
                        },
                    ]}
                    onChange={handleRefundUserMoneyChange}
                />
            </div>

            <div className={b('Option')}>
                Как возвращаем деньги?
                <RadioButton
                    value={moneyRefundType}
                    options={moneyRefundTypeOptions}
                    onChange={handleMoneyRefundTypeChange}
                />
            </div>

            <div className={b('Row')}>
                <div className={b('Option')}>
                    Сумма возврата, ₽
                    <MoneyInput
                        disabled={moneyRefundType !== EMoneyRefundTypes.AMOUNT}
                        value={
                            moneyRefundType === EMoneyRefundTypes.AMOUNT
                                ? moneyRefundAmountValue
                                : refundMoney
                        }
                        onChange={handleMoneyRefundAmountValueChange}
                    />
                </div>

                {hasDiscount && (
                    <div className={b('Option')}>
                        Пользователь получит, ₽
                        <MoneyInput
                            disabled
                            value={partialRefundAmount}
                            onChange={noop}
                        />
                    </div>
                )}
            </div>

            <div className={b('Option')}>
                Причина возврата
                <Textarea
                    value={reason}
                    onChange={handleReasonChange}
                    maxLength={500}
                />
            </div>

            <ActionButton
                progress={isLoading}
                disabled={isLoading || !isRefundOptionsValid || !reason.length}
                onClick={handleClick}
            >
                Оформить возврат денежных средств
            </ActionButton>
        </div>
    );
};

export default HotelMoneyOnlyRefundForm;
