import React, {useCallback, useEffect, useState} from 'react';
import {useDispatch, useSelector} from 'react-redux';
import {useToasts} from 'react-toast-notifications';
import {
    orderRequest,
    refundHotelMoneyOnlyRequest,
} from 'redux/reducers/order/actions';
import {block} from 'bem-cn';

import {ICalculatedHotelMoneyOnlyRefund, IStore} from 'redux/reducers/types';

import HotelMoneyOnlyRefundDeveloperForm from 'components/Order/HotelMoneyOnlyRefund/HotelMoneyOnlyRefundForm/HotelMoneyOnlyRefundDeveloperForm/HotelMoneyOnlyRefundDeveloperForm';
import ConfirmModal from 'components/Order/ConfirmModal/ConfirmModal';

const b = block('HotelMoneyOnlyRefundForm');

interface IOwnProps {
    className?: string;
    orderId: string;
    adminActionToken: string;
    calculatedRefund: ICalculatedHotelMoneyOnlyRefund;
}

const HotelMoneyOnlyRefundForm: React.FC<IOwnProps> = props => {
    const {className, orderId, adminActionToken, calculatedRefund} = props;

    const {isLoading, isFetched, error} = useSelector(
        (state: IStore) => state.order.refundHotelMoneyOnly,
    );

    const dispatch = useDispatch();
    const {addToast} = useToasts();

    const [isConfirmModalOpen, setConfirmModalOpen] = useState(false);
    const [moneyRefundAmountValue, setMoneyRefundAmountValue] = useState(1);
    const [isGenerateFinEvent, setGenerateFinEvent] = useState<boolean | null>(
        null,
    );
    const [isRefundUserMoney, setRefundUserMoney] = useState<boolean | null>(
        null,
    );
    const [isNeedRefresh, setNeedRefresh] = useState(false);
    const [confirmText, setConfirmText] = useState<React.ReactNode | null>(
        null,
    );
    const [reason, setReason] = useState('');

    useEffect(() => {
        if (!isNeedRefresh) {
            return;
        }

        setTimeout(
            () =>
                dispatch(
                    orderRequest({orderId, needToFetchPersonalData: false}),
                ),
            2000,
        );
        setNeedRefresh(false);
    }, [dispatch, orderId, isNeedRefresh]);

    const handleConfirm = useCallback(() => {
        dispatch(
            refundHotelMoneyOnlyRequest({
                orderId: orderId,
                adminActionToken: adminActionToken,
                generateFinEvents: Boolean(isGenerateFinEvent),
                refundUserMoney: Boolean(isRefundUserMoney),
                refundAmount: {
                    value: moneyRefundAmountValue,
                    currency: calculatedRefund.total_amount.currency,
                },
                newInvoiceAmount: {
                    value:
                        calculatedRefund.remaining_amount.value -
                        moneyRefundAmountValue,
                    currency: calculatedRefund.total_amount.currency,
                },
                reason,
                addToast,
            }),
        );
        setNeedRefresh(true);
    }, [
        orderId,
        adminActionToken,
        isGenerateFinEvent,
        isRefundUserMoney,
        moneyRefundAmountValue,
        calculatedRefund,
        reason,
        addToast,
        dispatch,
    ]);

    const handleActionByForm = useCallback(
        (
            requestedMoneyValue: number,
            requestedFinEvent: boolean,
            requestedRefundUserMoney: boolean,
            requestedReason: string,
            requestedConfirmText: React.ReactNode,
        ) => {
            setMoneyRefundAmountValue(requestedMoneyValue);
            setGenerateFinEvent(requestedFinEvent);
            setRefundUserMoney(requestedRefundUserMoney);
            setConfirmText(requestedConfirmText);
            setReason(requestedReason);
            setConfirmModalOpen(true);
        },
        [],
    );
    const handleCloseModal = useCallback(() => setConfirmModalOpen(false), []);

    return (
        <div className={b.mix(className)}>
            <HotelMoneyOnlyRefundDeveloperForm
                calculatedRefund={calculatedRefund}
                onAction={handleActionByForm}
                isLoading={isLoading}
            />

            <ConfirmModal
                isVisible={isConfirmModalOpen}
                onConfirm={handleConfirm}
                requestState={{isFetched, isLoading, hasError: Boolean(error)}}
                onClose={handleCloseModal}
                title="Будет оформлен возврат денег"
                successText="Возврат денег запущен успешно"
                errorText="Не удалось вернуть деньги"
            >
                {confirmText}
            </ConfirmModal>
        </div>
    );
};

export default HotelMoneyOnlyRefundForm;
