import React, {useCallback, useMemo, useState} from 'react';
import {useSelector} from 'react-redux';
import {block} from 'bem-cn';
import {noop} from 'lodash';

import {ICalculatedHotelOrderRefund, IStore} from 'redux/reducers/types';

import {usePartialRefundAmount} from 'components/Order/helpers/usePartialRefundAmount';

import MoneyInput from 'components/basic/MoneyInput/MoneyInput';
import ActionButton from 'components/lego/ActionButton/ActionButton';
import RadioButton from 'components/lego/RadioButton/RadioButton';
import Textarea from 'components/lego/Textarea/Textarea';

import './index.scss';

const b = block('HotelOrderRefundDeveloperForm');

interface IOwnProps {
    className?: string;
    calculatedRefund: ICalculatedHotelOrderRefund;
    onAction: (
        requestedMoneyValue: number,
        requestedFinEvent: boolean,
        reason: string,
        requestedConfirmText: React.ReactNode,
    ) => void;
    isLoading: boolean;
}

enum EGenerateFinEventsRadioButtonValues {
    YES = 'Yes',
    NO = 'No',
}

enum EMoneyRefundTypes {
    FULL = 'full',
    AMOUNT = 'amount',
    ONLY_CANCEL = 'only-cancel',
}

const HotelOrderRefundForm: React.FC<IOwnProps> = props => {
    const {className, calculatedRefund, onAction} = props;

    const {isLoading} = useSelector(
        (state: IStore) => state.order.refundHotelOrder,
    );

    const [isGenerateFinEvent, setGenerateFinEvent] = useState<boolean | null>(
        null,
    );
    const [moneyRefundAmountValue, setMoneyRefundAmountValue] = useState(1);
    const [moneyRefundType, setMoneyRefundType] = useState(
        EMoneyRefundTypes.FULL,
    );
    const [isMoneyRefundAmountValueValid, setMoneyRefundAmountValueValid] =
        useState(false);
    const [reason, setReason] = useState('');

    let refundMoney = calculatedRefund.paid_amount.value;

    const {partialRefundAmount, hasDiscount} = usePartialRefundAmount({
        moneyRefundType,
        inputValue: moneyRefundAmountValue,
    });

    if (moneyRefundType === EMoneyRefundTypes.AMOUNT) {
        refundMoney = partialRefundAmount;
    } else if (moneyRefundType === EMoneyRefundTypes.ONLY_CANCEL) {
        refundMoney = 0;
    }

    const handleGenerateFinEventChange = useCallback(event => {
        setGenerateFinEvent(
            event.target.value === EGenerateFinEventsRadioButtonValues.YES,
        );
    }, []);
    const handleMoneyRefundAmountValueChange = useCallback(
        (isValid: boolean, value: number) => {
            if (isValid) {
                setMoneyRefundAmountValue(value);
            }

            setMoneyRefundAmountValueValid(isValid);
        },
        [],
    );
    const handleMoneyRefundTypeChange = useCallback(event => {
        setMoneyRefundType(event.target.value);
        setMoneyRefundAmountValueValid(true);
        setMoneyRefundAmountValue(1);
    }, []);

    const isRefundOptionsValid =
        isGenerateFinEvent !== null &&
        (moneyRefundType !== EMoneyRefundTypes.AMOUNT ||
            isMoneyRefundAmountValueValid);

    const generateFinEventRadioButtonValue = isGenerateFinEvent
        ? EGenerateFinEventsRadioButtonValues.YES
        : EGenerateFinEventsRadioButtonValues.NO;

    const hotelierRefundOptions = useMemo(
        () => [
            {value: EGenerateFinEventsRadioButtonValues.YES, children: 'Да'},
            {value: EGenerateFinEventsRadioButtonValues.NO, children: 'Нет'},
        ],
        [],
    );
    const moneyRefundTypeOptions = [
        {value: EMoneyRefundTypes.FULL, children: 'Полностью'},
        {value: EMoneyRefundTypes.AMOUNT, children: 'Частично'},
        {value: EMoneyRefundTypes.ONLY_CANCEL, children: 'Только отмена брони'},
    ];

    const handleClick = useCallback(() => {
        onAction(
            refundMoney,
            Boolean(isGenerateFinEvent),
            reason,
            <React.Fragment>
                <p>
                    Пользователю будет возвращено{' '}
                    <span className={b('Bold')}>{refundMoney} руб</span> из{' '}
                    <span className={b('Bold')}>
                        {calculatedRefund.paid_amount.value} руб
                    </span>{' '}
                    уплаченных (по правилам возврата должно быть{' '}
                    <span className={b('Bold')}>
                        {calculatedRefund.refund_amount_by_rules.value} руб
                    </span>
                    ), при общей сумме{' '}
                    <span className={b('Bold')}>
                        {calculatedRefund.total_amount.value} руб
                    </span>
                </p>
                <p>
                    Финансовые события{' '}
                    {isGenerateFinEvent ? 'будут' : 'не будут'} сгенерированы
                </p>
            </React.Fragment>,
        );
    }, [calculatedRefund, onAction, refundMoney, isGenerateFinEvent, reason]);

    const handleReasonChange = useCallback(
        event => setReason(event.target.value),
        [],
    );

    return (
        <div className={b.mix(className)}>
            <div className={b('Option')}>
                Вернуть деньги отельера?
                <RadioButton
                    value={
                        isGenerateFinEvent === null
                            ? ''
                            : generateFinEventRadioButtonValue
                    }
                    options={hotelierRefundOptions}
                    onChange={handleGenerateFinEventChange}
                />
            </div>

            <div className={b('Option')}>
                Как возвращаем деньги?
                <RadioButton
                    value={moneyRefundType}
                    options={moneyRefundTypeOptions}
                    onChange={handleMoneyRefundTypeChange}
                />
            </div>

            <div className={b('Row')}>
                <div className={b('Option')}>
                    Сумма возврата, ₽
                    <MoneyInput
                        disabled={moneyRefundType !== EMoneyRefundTypes.AMOUNT}
                        value={
                            moneyRefundType === EMoneyRefundTypes.AMOUNT
                                ? moneyRefundAmountValue
                                : refundMoney
                        }
                        onChange={handleMoneyRefundAmountValueChange}
                    />
                </div>

                {hasDiscount && (
                    <div className={b('Option')}>
                        Пользователь получит, ₽
                        <MoneyInput
                            disabled
                            value={partialRefundAmount}
                            onChange={noop}
                        />
                    </div>
                )}
            </div>

            <div className={b('Option')}>
                Причина отмены
                <Textarea value={reason} onChange={handleReasonChange} />
            </div>

            <ActionButton
                progress={isLoading}
                disabled={isLoading || !isRefundOptionsValid || !reason.length}
                onClick={handleClick}
            >
                Оформить отмену заказа
            </ActionButton>
        </div>
    );
};

export default HotelOrderRefundForm;
