import React, {useCallback, useEffect, useState} from 'react';
import {useDispatch, useSelector} from 'react-redux';
import {useToasts} from 'react-toast-notifications';
import {
    orderRequest,
    refundHotelOrderRequest,
} from 'redux/reducers/order/actions';
import {block} from 'bem-cn';

import {ICalculatedHotelOrderRefund, IStore} from 'redux/reducers/types';

import HotelOrderRefundDeveloperForm from 'components/Order/HotelOrderRefund/HotelOrderRefundForm/HotelOrderRefundDeveloperForm/HotelOrderRefundDeveloperForm';
import ConfirmModal from 'components/Order/ConfirmModal/ConfirmModal';

const b = block('HotelOrderRefundForm');

interface IOwnProps {
    className?: string;
    orderId: string;
    adminActionToken: string;
    calculatedRefund: ICalculatedHotelOrderRefund;
}

const HotelOrderRefundForm: React.FC<IOwnProps> = props => {
    const {className, orderId, adminActionToken, calculatedRefund} = props;

    const {isLoading, isFetched, error} = useSelector(
        (state: IStore) => state.order.refundHotelOrder,
    );

    const dispatch = useDispatch();
    const {addToast} = useToasts();

    const [isConfirmModalOpen, setConfirmModalOpen] = useState(false);
    const [moneyRefundAmountValue, setMoneyRefundAmountValue] = useState(1);
    const [isGenerateFinEvent, setGenerateFinEvent] = useState<boolean | null>(
        null,
    );
    const [isNeedRefresh, setNeedRefresh] = useState(false);
    const [confirmText, setConfirmText] = useState<React.ReactNode | null>(
        null,
    );
    const [reason, setReason] = useState('');

    useEffect(() => {
        if (!isNeedRefresh) {
            return;
        }

        setTimeout(
            () =>
                dispatch(
                    orderRequest({orderId, needToFetchPersonalData: false}),
                ),
            2000,
        );
        setNeedRefresh(false);
    }, [dispatch, orderId, isNeedRefresh]);

    const handleConfirm = useCallback(() => {
        dispatch(
            refundHotelOrderRequest({
                orderId: orderId,
                adminActionToken: adminActionToken,
                generateFinEvents: Boolean(isGenerateFinEvent),
                refundAmount: {
                    value: moneyRefundAmountValue,
                    currency: calculatedRefund.total_amount.currency,
                },
                reason,
                addToast,
            }),
        );
        setNeedRefresh(true);
    }, [
        orderId,
        adminActionToken,
        isGenerateFinEvent,
        moneyRefundAmountValue,
        calculatedRefund,
        reason,
        addToast,
        dispatch,
    ]);

    const handleActionByForm = useCallback(
        (
            requestedMoneyValue: number,
            requestedFinEvent: boolean,
            requestedReason: string,
            requestedConfirmText: React.ReactNode,
        ) => {
            setMoneyRefundAmountValue(requestedMoneyValue);
            setGenerateFinEvent(requestedFinEvent);
            setConfirmText(requestedConfirmText);
            setReason(requestedReason);
            setConfirmModalOpen(true);
        },
        [],
    );
    const handleCloseModal = useCallback(() => setConfirmModalOpen(false), []);

    const form = (
        <HotelOrderRefundDeveloperForm
            calculatedRefund={calculatedRefund}
            onAction={handleActionByForm}
            isLoading={isLoading}
        />
    );

    return (
        <div className={b.mix(className)}>
            {form}

            <ConfirmModal
                isVisible={isConfirmModalOpen}
                onConfirm={handleConfirm}
                onClose={handleCloseModal}
                requestState={{
                    isFetched: isFetched,
                    isLoading: isLoading,
                    hasError: Boolean(error),
                }}
                title="Будет оформлен возврат заказа"
                successText="Возврат заказа запущен успешно"
                errorText="Не удалось отменить заказ"
            >
                {confirmText}
            </ConfirmModal>
        </div>
    );
};

export default HotelOrderRefundForm;
