import React, {useEffect, useMemo} from 'react';
import {useSelector, useDispatch} from 'react-redux';
import {useRouteMatch} from 'react-router-dom';
import {Loader} from '@yandex-data-ui/common';
import {block} from 'bem-cn';
import {orderRequest} from 'redux/reducers/order/actions';

import {IStore} from 'redux/reducers/types';

import getHumanMoscowFormat from 'lib/time/getHumanMoscowFormat';
import getHumanFormatWithTime from 'lib/time/getHumanFormatWithTime';
import getPersonalFieldText from 'lib/getPersonalFieldText';
import {getOrderPartnerText} from 'lib/dictionaries/orderPartner';

import Heading from 'components/basic/Heading/Heading';
import NotFound from 'components/NotFound/NotFound';
import OrderMainInfo from 'components/Order/OrderMainInfo/OrderMainInfo';
import OrderStartrekTickets from 'components/Order/OrderStartrekTickets/OrderStartrekTickets';
import HotelCancellationInfo from 'components/Order/HotelCancellationInfo/HotelCancellationInfo';
import OrderCostInfo from 'components/Order/OrderCostInfo/OrderCostInfo';
import OrderAccess from 'components/Order/OrderAccess/OrderAccess';
import OrderContacts from 'components/Order/OrderContacts/OrderContacts';
import OrderNotifications from 'components/Order/OrderNotifications/OrderNotifications';
import OrderGetPersonalInfo from 'components/Order/OrderGetPersonalInfo/OrderGetPersonalInfo';
import OrderRefund from 'components/Order/OrderRefund/OrderRefund';
import OrderRetryMoneyRefund from 'components/Order/OrderRetryMoneyRefund/OrderRetryMoneyRefund';
import OrderTrainInfo from 'components/Order/OrderTrainInfo/OrderTrainInfo';
import OrderPassengers from 'components/Order/OrderPassengers/OrderPassengers';
import OrderBusInfo from 'components/Order/OrderBusInfo/OrderBusInfo';
import OrderBusPassengers from 'components/Order/OrderBusPassengers/OrderBusPassengers';
import OrderRoomInfo from 'components/Order/OrderRoomInfo/OrderRoomInfo';
import OrderVouchers from 'components/Order/OrderVouchers/OrderVouchers';
import OrderRaw from 'components/Order/OrderRaw/OrderRaw';
import OrderPaymentsAndRefunds from 'components/Order/OrderPaymentsAndRefunds/OrderPaymentsAndRefunds';
import OrderTaxiPromo from 'components/Order/OrderTaxiPromo/OrderTaxiPromo';
import OrderMirPromo from 'components/Order/OrderMirPromo/OrderMirPromo';
import OrderDeferredPayment from 'components/Order/OrderDeferredPayment/OrderDeferredPayment';
import OrderPostPay from 'components/Order/OrderPostPay/OrderPostPay';
import OrderAdminActions from 'components/Order/OrderAdminActions/OrderAdminActions';
import OrderShotInfo from 'components/Order/OrderShotInfo/OrderShotInfo';
import ClipboardButton from 'components/basic/ClipboardButton/ClipboardButton';
import HotelOrderRefund from 'components/Order/HotelOrderRefund/HotelOrderRefund';
import YandexPlusInfo from 'components/Order/YandexPlusInfo/YandexPlusInfo';
import HotelMoneyOnlyRefund from 'components/Order/HotelMoneyOnlyRefund/HotelMoneyOnlyRefund';
import HotelGuestsInfo from 'components/Order/HotelGuestsInfo/HotelGuestInfo';

import './index.scss';

const b = block('Order');

interface IRouterParams {
    id: string;
}

const Order: React.FC = () => {
    const match = useRouteMatch<IRouterParams>();

    const orderId = match?.params.id;

    const dispatch = useDispatch();

    const {
        env,
        order: {
            item: {value: order, isLoading, isFetched, error},
            refund: {isLoading: refundIsLoading},
            startrekTickets: {
                value: startrekTickets,
                isLoading: startrekTicketsIsLoading,
                isFetched: startrekTicketsIsFetched,
                error: startrekTicketsError,
            },
        },
        settings,
    } = useSelector((state: IStore) => state);

    useEffect(() => {
        if (!orderId || order?.id === orderId) {
            return;
        }

        dispatch(orderRequest({orderId, needToFetchPersonalData: false}));
    }, [dispatch, order, orderId]);

    const content = useMemo(() => {
        if (!isLoading && !isFetched) {
            return null;
        }

        if (isLoading) {
            return (
                <div className={b('loaderWrapper')}>
                    <Loader size="m" />
                </div>
            );
        }

        if (error) {
            if (error.status === 404) {
                return <NotFound />;
            }

            return <div className="text-danger">Ошибка загрузки заказа</div>;
        }

        if (!order) {
            return null;
        }

        const {
            status,
            flags,
            hotel_order_items,
            avia_order_items,
            train_order_items,
            bus_order_items,
            payment_info,
            star_trek_ticket_ids,
            promo_campaigns,
            order_price_info,
            vouchers,
            payment_schedule,
            is_post_pay_eligible,
            is_post_paid,
        } = order;

        const hotelOrderItem = hotel_order_items[0] as
            | typeof hotel_order_items[0]
            | undefined;
        const aviaOrderItem = avia_order_items[0] as
            | typeof avia_order_items[0]
            | undefined;

        // Включает показ generic vouchers. В текущей реализации бэкенда нет ссылок. Убрать после TRAVELBACK-1690
        const enableVouchers = false;

        return (
            <div>
                <OrderShotInfo
                    order={order}
                    isTechTexts={settings.isTechTexts}
                />

                <OrderMainInfo
                    className={b('block')}
                    isTechTexts={settings.isTechTexts}
                    order={order}
                    hotelOrderItem={hotelOrderItem}
                    trainOrderItems={train_order_items}
                />

                {star_trek_ticket_ids && (
                    <OrderStartrekTickets
                        className={b('block')}
                        ticketsIds={star_trek_ticket_ids}
                        ticketsData={startrekTickets}
                        isLoading={startrekTicketsIsLoading}
                        isFetched={startrekTicketsIsFetched}
                        error={startrekTicketsError}
                    />
                )}

                {train_order_items &&
                    train_order_items.map((trainOrderSegment, index) => (
                        <>
                            <OrderTrainInfo
                                className={b('block')}
                                orderStatus={status}
                                trainOrderItem={trainOrderSegment}
                                hasMaskedInfo={flags.has_masked_info}
                                segmentIndex={
                                    train_order_items.length > 1 ? index : null
                                }
                            />

                            <OrderPassengers
                                className={b('block')}
                                trainOrderItem={trainOrderSegment}
                                hasMaskedInfo={flags.has_masked_info}
                            />
                        </>
                    ))}

                {(hotelOrderItem || aviaOrderItem) && (
                    <OrderCostInfo
                        className={b('block')}
                        orderPriceInfo={order_price_info}
                    />
                )}

                {hotelOrderItem && (
                    <OrderPostPay
                        className={b('block')}
                        eligible={is_post_pay_eligible}
                        used={is_post_paid}
                    />
                )}

                {hotelOrderItem && (
                    <OrderDeferredPayment
                        className={b('block')}
                        paymentSchedule={payment_schedule}
                    />
                )}

                <div className={b('block')}>
                    <Heading level="4">Партнер</Heading>
                    <div>
                        <div>
                            Партнер:{' '}
                            {order.partners
                                .map(partner =>
                                    getOrderPartnerText(
                                        partner,
                                        settings.isTechTexts,
                                    ),
                                )
                                .join(', ')}
                        </div>
                        {hotelOrderItem && (
                            <div>
                                <div>
                                    ID партнера:{' '}
                                    {hotelOrderItem.itinerary_id || 'нет'}
                                </div>
                                <div>
                                    Номер заказа:{' '}
                                    {
                                        hotelOrderItem.confirmation_info
                                            .confirmation_id
                                    }
                                </div>
                            </div>
                        )}
                    </div>
                </div>

                {bus_order_items &&
                    bus_order_items.map(busOrder => (
                        <>
                            <OrderBusInfo
                                className={b('block')}
                                busOrderItem={busOrder}
                                hasMaskedInfo={flags.has_masked_info}
                            />

                            <OrderBusPassengers
                                className={b('block')}
                                busOrderItem={busOrder}
                                hasMaskedInfo={flags.has_masked_info}
                            />
                        </>
                    ))}

                {aviaOrderItem && (
                    <div className={b('block')}>
                        <Heading level="4">Персональные данные</Heading>

                        <div>
                            {getPersonalFieldText(
                                aviaOrderItem.passengers &&
                                    aviaOrderItem.passengers.map(
                                        (
                                            {
                                                first_name,
                                                last_name,
                                                middle_name,
                                                nationality,
                                                document_type,
                                                document_number,
                                                date_of_birth,
                                            },
                                            index,
                                        ) => (
                                            <div
                                                key={index}
                                                className={b('personalData')}
                                            >
                                                <div>
                                                    ФИО: {first_name}{' '}
                                                    {last_name} {middle_name}
                                                </div>
                                                <div>
                                                    Гражданство: {nationality}
                                                </div>
                                                <div>
                                                    Тип документа:{' '}
                                                    {document_type}
                                                </div>
                                                <div>
                                                    Номер документа:{' '}
                                                    {document_number}
                                                </div>
                                                <div>
                                                    Дата рождения:{' '}
                                                    {date_of_birth}
                                                </div>
                                            </div>
                                        ),
                                    ),
                                order.flags.has_masked_info,
                            )}
                        </div>
                    </div>
                )}

                <OrderPaymentsAndRefunds
                    className={b('block')}
                    paymentInfo={payment_info}
                    settings={settings}
                    env={env}
                />

                <OrderContacts className={b('block')} order={order} />

                {(train_order_items || hotelOrderItem) && (
                    <OrderRefund
                        className={b('block')}
                        order={order}
                        isLoading={refundIsLoading}
                    />
                )}

                {(train_order_items || bus_order_items) && (
                    <OrderRetryMoneyRefund
                        className={b('block')}
                        order={order}
                    />
                )}

                <OrderAccess
                    className={b('block')}
                    authorizedUsers={order.authorized_users}
                />

                {hotelOrderItem && (
                    <div className={b('block')}>
                        <Heading level="4">Бронь отеля</Heading>

                        <div>
                            Дата и время подтверждения:{' '}
                            {hotelOrderItem.confirmation_info.confirmed_at
                                ? getHumanMoscowFormat(
                                      hotelOrderItem.confirmation_info
                                          .confirmed_at,
                                  )
                                : 'нет'}
                        </div>

                        <div>
                            Дата и время отмены:{' '}
                            {hotelOrderItem.refund_info &&
                            hotelOrderItem.refund_info.refund_date_time
                                ? getHumanMoscowFormat(
                                      hotelOrderItem.refund_info
                                          .refund_date_time,
                                  )
                                : 'нет'}
                        </div>
                        <div>
                            Подтверждающий документ:{' '}
                            {getPersonalFieldText(
                                hotelOrderItem.confirmation_info.document_url &&
                                    !order.flags.has_masked_info && (
                                        <a
                                            href={
                                                hotelOrderItem.confirmation_info
                                                    .document_url
                                            }
                                            target="_blank"
                                            rel="noopener noreferrer"
                                        >
                                            {
                                                hotelOrderItem.confirmation_info
                                                    .document_url
                                            }
                                        </a>
                                    ),
                                order.flags.has_masked_info,
                            )}
                        </div>
                        {enableVouchers && vouchers && (
                            <OrderVouchers
                                vouchers={vouchers}
                                hasMaskedData={order.flags.has_masked_info}
                                isTechTexts={settings.isTechTexts}
                            />
                        )}
                        {hotelOrderItem.room_info && (
                            <OrderRoomInfo
                                roomInfo={hotelOrderItem.room_info}
                                isTechTexts={settings.isTechTexts}
                            />
                        )}
                    </div>
                )}

                {hotelOrderItem && (
                    <React.Fragment>
                        <div className={b('block')}>
                            <Heading level="4">Отель</Heading>
                            <div>
                                Отель:{' '}
                                {hotelOrderItem.basic_hotel_info.name || 'нет'}
                            </div>
                            <div>
                                Адрес:{' '}
                                {hotelOrderItem.basic_hotel_info.address ||
                                    'нет'}
                            </div>
                            <div>
                                Телефон:{' '}
                                {hotelOrderItem.basic_hotel_info.phone || 'нет'}
                            </div>

                            <div className={b('cancellationInfo')}>
                                Условия отмены бронирования
                                <HotelCancellationInfo
                                    cancellationInfo={
                                        hotelOrderItem.cancellation_info
                                    }
                                />
                            </div>

                            <div>
                                <a
                                    href="https://yandex.ru/legal/hotels_booking_conditions/#index__section_bzg_4bw_qhb"
                                    target="_blank"
                                    rel="noopener noreferrer"
                                >
                                    Общие условия возврата
                                </a>
                            </div>

                            <div>
                                Permalink:{' '}
                                {hotelOrderItem.basic_hotel_info.permalink}{' '}
                                <a
                                    href={
                                        hotelOrderItem.basic_hotel_info
                                            .altay_permaling_url
                                    }
                                    target="_blank"
                                    rel="noopener noreferrer"
                                >
                                    Altay
                                </a>
                            </div>

                            <div>
                                Original Id:{' '}
                                {hotelOrderItem.basic_hotel_info.original_id ||
                                    'нет'}{' '}
                                {hotelOrderItem.basic_hotel_info
                                    .altay_signals_url && (
                                    <a
                                        href={
                                            hotelOrderItem.basic_hotel_info
                                                .altay_signals_url
                                        }
                                        target="_blank"
                                        rel="noopener noreferrer"
                                    >
                                        Altay
                                    </a>
                                )}
                            </div>
                        </div>

                        {order.flags.can_modify_hotel_order_details && (
                            <HotelGuestsInfo
                                orderId={order.id}
                                adminActionToken={order.admin_action_token}
                                guests={hotelOrderItem.guest_info?.guests}
                                className={b('block')}
                            />
                        )}

                        {order.flags.can_refund_hotel_order && (
                            <HotelOrderRefund
                                orderId={order.id}
                                adminActionToken={order.admin_action_token}
                                className={b('block')}
                            />
                        )}

                        {order.flags.can_refund_hotel_money_only && (
                            <HotelMoneyOnlyRefund
                                orderId={order.id}
                                adminActionToken={order.admin_action_token}
                                className={b('block')}
                            />
                        )}
                    </React.Fragment>
                )}

                {aviaOrderItem && (
                    <div className={b('block')}>
                        <Heading level="4">Рейс самолета</Heading>

                        <div>
                            Номер для онлайн-регистрации на рейс:{' '}
                            {aviaOrderItem.pnr || 'нет'}
                        </div>

                        {aviaOrderItem.legs.map(({segments}, index) => (
                            <div key={index} className={b('leg')}>
                                {segments.map(
                                    (
                                        {
                                            flight_number,
                                            departure_airport,
                                            arrival_airport,
                                            departure_at,
                                            arrival_at,
                                        },
                                        index,
                                    ) => (
                                        <div
                                            key={index}
                                            className={b('flight')}
                                        >
                                            <div>
                                                Номер рейса: {flight_number}
                                            </div>
                                            <div>
                                                Направление: {departure_airport}{' '}
                                                - {arrival_airport}
                                            </div>
                                            <div>
                                                Дата и время вылета:{' '}
                                                {getHumanFormatWithTime(
                                                    departure_at,
                                                )}
                                            </div>
                                            <div>
                                                Дата и время прилёта:{' '}
                                                {getHumanFormatWithTime(
                                                    arrival_at,
                                                )}
                                            </div>
                                        </div>
                                    ),
                                )}
                            </div>
                        ))}
                    </div>
                )}

                {promo_campaigns?.taxi2020 && (
                    <OrderTaxiPromo
                        className={b('block')}
                        promo={promo_campaigns.taxi2020}
                    />
                )}

                {hotelOrderItem && promo_campaigns?.mir2020 && (
                    <OrderMirPromo
                        className={b('block')}
                        promo={promo_campaigns.mir2020}
                        isPostPay={is_post_paid}
                    />
                )}

                {promo_campaigns?.yandex_plus?.mode && (
                    <YandexPlusInfo
                        className={b('block')}
                        data={promo_campaigns?.yandex_plus}
                        isTechTexts={settings.isTechTexts}
                        env={env}
                    />
                )}

                {(hotelOrderItem || train_order_items) && (
                    <OrderNotifications order={order} className={b('block')} />
                )}

                <OrderAdminActions order={order} className={b('block')} />

                <OrderRaw className={b('block')} order={order} />
            </div>
        );
    }, [
        isLoading,
        order,
        settings,
        refundIsLoading,
        error,
        isFetched,
        startrekTickets,
        startrekTicketsError,
        startrekTicketsIsFetched,
        startrekTicketsIsLoading,
        env,
    ]);

    return (
        <div className={b()}>
            <div className={b('titleWrapper')}>
                <Heading className={b('title')} level="1">
                    {order?.broken && <span title="Сломанный заказ">⚠️</span>}{' '}
                    Заказ {order?.yandex_order_id}
                </Heading>

                {Boolean(order) && (
                    <ClipboardButton
                        text={order?.yandex_order_id || ''}
                        size="m"
                    />
                )}

                <OrderGetPersonalInfo
                    className={b('getPersonalInfo').toString()}
                    theme="action"
                />
            </div>

            {content}
        </div>
    );
};

export default Order;
