import React, {useMemo} from 'react';
import moment, {Moment} from 'moment';
import {block} from 'bem-cn';

import {IBusOrderItem} from 'redux/reducers/types/bus';

import {BUS_ORDER_DOCUMENT_TYPE_DICT} from 'lib/dictionaries/busOrderDocumentType';
import {BUS_ORDER_TICKET_TYPE_DICT} from 'lib/dictionaries/busOrderTicketType';
import getPersonalFieldText from 'lib/getPersonalFieldText';

import Heading from 'components/basic/Heading/Heading';

import './index.scss';

const b = block('OrderBusPassengers');

interface IOrderBusPassengersProps {
    className?: string;
    busOrderItem: IBusOrderItem;
    hasMaskedInfo: boolean;
}

const OrderBusPassengers: React.FC<IOrderBusPassengersProps> = props => {
    const {
        className,
        busOrderItem: {passengers},
        hasMaskedInfo,
    } = props;

    const now = useMemo((): Moment => moment(), []);

    return (
        <div className={b.mix(className)}>
            <Heading level="4">Пассажиры</Heading>

            <div className={b('passengers')}>
                {passengers.map(
                    (
                        {
                            first_name,
                            middle_name,
                            last_name,
                            birthday,
                            document_type,
                            document_number,
                            ticket_type,
                            seat_id,
                            blank_url,
                        },
                        index,
                    ) => {
                        return (
                            <div className={b('passenger')} key={index}>
                                {getPersonalFieldText(
                                    first_name,
                                    hasMaskedInfo,
                                )}{' '}
                                {getPersonalFieldText(
                                    middle_name,
                                    hasMaskedInfo,
                                )}{' '}
                                {getPersonalFieldText(last_name, hasMaskedInfo)}{' '}
                                (
                                {getPersonalFieldText(
                                    birthday ? now.diff(birthday, 'year') : '',
                                    hasMaskedInfo,
                                )}
                                )
                                <br />
                                {
                                    BUS_ORDER_DOCUMENT_TYPE_DICT[document_type]
                                }{' '}
                                {getPersonalFieldText(
                                    document_number,
                                    hasMaskedInfo,
                                )}
                                <br />
                                {BUS_ORDER_TICKET_TYPE_DICT[ticket_type]}
                                {' 💺 '}
                                {seat_id}
                                <br />
                                {blank_url && (
                                    <a
                                        href={blank_url}
                                        target="_blank"
                                        rel="noreferrer"
                                    >
                                        Билет
                                    </a>
                                )}
                            </div>
                        );
                    },
                )}
            </div>
        </div>
    );
};

export default OrderBusPassengers;
