import React, {useState, useCallback, useEffect} from 'react';
import {useDispatch, useSelector} from 'react-redux';
import {useToasts} from 'react-toast-notifications';
import {TextInput} from 'lego-on-react';
import {Loader} from '@yandex-data-ui/common';
import {block} from 'bem-cn';
import {
    orderEditEmailRequest,
    resetOrderEditEmail,
} from 'redux/reducers/order/actions';

import {IOrderInfo, IStore} from 'redux/reducers/types';

import getPersonalFieldText from 'lib/getPersonalFieldText';

import './index.scss';

import ClipboardButton from 'components/basic/ClipboardButton/ClipboardButton';
import ActionButton from 'components/lego/ActionButton/ActionButton';

const b = block('OrderContactsEmail');

interface IOrderContactsEmailProps {
    order: IOrderInfo;
}

const OrderContactsEmail: React.FC<IOrderContactsEmailProps> = props => {
    const {
        order: {
            id: orderId,
            owner,
            flags: {has_masked_info},
        },
    } = props;

    const {addToast} = useToasts();

    const dispatch = useDispatch();

    const {
        order: {
            emailEdition: {isLoading, isFetched},
        },
    } = useSelector((store: IStore) => store);

    const [isEditable, setIsEditable] = useState(false);
    const [email, setEmail] = useState(
        String(getPersonalFieldText(owner && owner.email, has_masked_info)),
    );

    const handleChange = useCallback((newEmail: string) => {
        setEmail(newEmail);
    }, []);

    const handleClick = useCallback(() => {
        if (!isEditable) {
            setIsEditable(true);

            return;
        }

        dispatch(orderEditEmailRequest({orderId, newEmail: email, addToast}));
    }, [orderId, email, isEditable]);

    useEffect(() => {
        if (isFetched) {
            setIsEditable(false);
        }
    }, [isFetched]);

    useEffect(() => {
        return () => {
            dispatch(resetOrderEditEmail());
        };
    }, []);

    return (
        <div className={b()}>
            Почта{has_masked_info && `: ${email}`}
            {!has_masked_info && (
                <div className={b('inputAndButtonWrapper')}>
                    <TextInput
                        cls={b('input')}
                        theme="normal"
                        size="m"
                        view="default"
                        tone="default"
                        text={email}
                        hasClear
                        disabled={isLoading || !isEditable}
                        onChange={handleChange}
                    />

                    <ClipboardButton text={email} size="s" />

                    {isLoading ? (
                        <Loader size="m" className={b('loader').toString()} />
                    ) : (
                        <ActionButton
                            className={b('button').toString()}
                            type="submit"
                            onClick={handleClick}
                        >
                            {isEditable ? 'Сохранить' : 'Изменить'}
                        </ActionButton>
                    )}
                </div>
            )}
        </div>
    );
};

export default OrderContactsEmail;
