import React, {useState, useCallback, useEffect} from 'react';
import {useDispatch, useSelector} from 'react-redux';
import {useToasts} from 'react-toast-notifications';
import {TextInput} from 'lego-on-react';
import {Loader} from '@yandex-data-ui/common';
import {block} from 'bem-cn';
import {
    orderEditPhoneRequest,
    resetOrderEditPhone,
} from 'redux/reducers/order/actions';

import {IOrderInfo, IStore} from 'redux/reducers/types';

import getPersonalFieldText from 'lib/getPersonalFieldText';

import ActionButton from 'components/lego/ActionButton/ActionButton';

import './index.scss';

const b = block('OrderContactsPhone');

interface IOrderContactsPhoneProps {
    order: IOrderInfo;
}

const OrderContactsPhone: React.FC<IOrderContactsPhoneProps> = props => {
    const {
        order: {
            id: orderId,
            owner,
            flags: {has_masked_info},
        },
    } = props;

    const {addToast} = useToasts();

    const dispatch = useDispatch();

    const {
        order: {
            phoneEdition: {isLoading, isFetched},
        },
    } = useSelector((store: IStore) => store);

    const [isEditable, setIsEditable] = useState(false);
    const [phone, setPhone] = useState(
        String(getPersonalFieldText(owner && owner.phone, has_masked_info)),
    );

    const handleChange = useCallback((newPhone: string) => {
        setPhone(newPhone);
    }, []);

    const handleClick = useCallback(() => {
        if (!isEditable) {
            setIsEditable(true);

            return;
        }

        dispatch(orderEditPhoneRequest({orderId, newPhone: phone, addToast}));
    }, [orderId, phone, isEditable]);

    useEffect(() => {
        if (isFetched) {
            setIsEditable(false);
        }
    }, [isFetched]);

    useEffect(() => {
        return () => {
            dispatch(resetOrderEditPhone());
        };
    }, []);

    return (
        <div className={b()}>
            Телефон{has_masked_info && `: ${phone}`}
            {!has_masked_info && (
                <div className={b('inputAndButtonWrapper')}>
                    <TextInput
                        cls={b('input')}
                        theme="normal"
                        size="m"
                        view="default"
                        tone="default"
                        text={phone}
                        hasClear
                        disabled={isLoading || !isEditable}
                        onChange={handleChange}
                    />

                    {isLoading ? (
                        <Loader size="m" className={b('loader').toString()} />
                    ) : (
                        <ActionButton
                            className={b('button').toString()}
                            type="submit"
                            onClick={handleClick}
                        >
                            {isEditable ? 'Сохранить' : 'Изменить'}
                        </ActionButton>
                    )}
                </div>
            )}
        </div>
    );
};

export default OrderContactsPhone;
