import React, {useCallback, useEffect, useMemo, useState} from 'react';
import {useSelector, useDispatch} from 'react-redux';
import {useRouteMatch, useLocation} from 'react-router-dom';
import {Loader} from '@yandex-data-ui/common';
import {block} from 'bem-cn';
import {
    orderLogsRequest,
    orderRequest,
    resetOrderLogs,
    resetOrder,
} from 'redux/reducers/order/actions';

import {DEFAULT_ORDER_LOGS_COUNT} from 'constants/search';

import {isOrderLogLevel, IStore} from 'redux/reducers/types';

import useBoolean from 'lib/hooks/useBoolean';
import parseOrderLogsUrl from 'lib/url/logs';
import getHumanMoscowFormatWithMilliseconds from 'lib/time/getHumanMoscowFormatWithMilliseconds';
import {extendUrl} from 'lib/url/utils';

import {ytLogsLink} from 'components/Order/OrderLogs/Logslink';
import Heading from 'components/basic/Heading/Heading';
import Pagination from 'components/basic/Pagination/Pagination';
import OrderLogsFilters from 'components/Order/OrderLogs/OrderLogsFilters/OrderLogsFilters';
import Modal from 'components/basic/Modal/Modal';

import './index.scss';

const b = block('OrderLogs');

const USER_ACTION_LOGGER_NAME = 'ru.yandex.travel.AdminUserActionLogger';

const LARGE_MESSAGE_MINIMAL_LENGTH = 350;

const getJSONLikeString = (text: string): string => {
    try {
        return JSON.stringify(JSON.parse(text), null, '\t');
    } catch {
        return text;
    }
};

interface IRouterParams {
    id: string;
}

const OrderLogs: React.FC = () => {
    const location = useLocation();

    const {search} = location;

    const match = useRouteMatch<IRouterParams>();

    const orderId = match?.params.id;

    const dispatch = useDispatch();

    const {
        env,
        order: {
            item: {value: order, isFetched: item_isFetched},
            logs: {value: logs, overallItemsCount, isLoading, isFetched, error},
        },
    } = useSelector((state: IStore) => state);

    const {
        value: modalIsVisible,
        setTrue: openModal,
        setFalse: hideModal,
    } = useBoolean(false);

    const [modalContent, setModalContent] = useState('');

    const {trimLarge} = parseOrderLogsUrl(search);

    useEffect(() => {
        if (!orderId) {
            return;
        }

        const {page, count, level, logger, message} = parseOrderLogsUrl(search);

        dispatch(
            orderLogsRequest({
                orderId,
                page: page !== undefined ? Number(page) - 1 : undefined,
                count: count !== undefined ? Number(count) : undefined,
                level: isOrderLogLevel(level) ? level : undefined,
                logger,
                message,
            }),
        );
        dispatch(orderRequest({orderId, needToFetchPersonalData: false}));
    }, [orderLogsRequest, orderRequest, orderId, search]);

    useEffect(
        () => () => {
            dispatch(resetOrderLogs());
        },
        [resetOrderLogs, dispatch],
    );

    useEffect(
        () => () => {
            dispatch(resetOrder());
        },
        [resetOrder, dispatch],
    );

    const pageLinkBuilder = useCallback(
        (page: number, count: number): string => {
            return extendUrl(location, {
                page: String(page),
                count: String(count),
            });
        },
        [location],
    );

    const handleLogClick = useCallback((logContext: string) => {
        setModalContent(logContext);

        openModal();
    }, []);

    const pagination = useMemo(() => {
        const {page = '1', count = `${DEFAULT_ORDER_LOGS_COUNT}`} =
            parseOrderLogsUrl(search);

        return (
            <Pagination
                className={b('pagination')}
                itemsCount={overallItemsCount}
                itemsPerPage={Number(count)}
                currentPage={Number(page)}
                buttonsLimit={5}
                pageLinkBuilder={pageLinkBuilder}
            />
        );
    }, [search, pageLinkBuilder, overallItemsCount]);

    const content = useMemo(() => {
        if (!isLoading && !isFetched) {
            return null;
        }

        if (isLoading) {
            return (
                <div className={b('loaderWrapper')}>
                    <Loader size="m" />
                </div>
            );
        }

        if (error) {
            return <div className="text-danger">Ошибка загрузки заказа</div>;
        }

        if (!logs.length) {
            return (
                <div>
                    Нет подходящих логов? Возможно, они были автоматически
                    удалены из YDB. Попробуйте
                    {item_isFetched ? (
                        <a href={ytLogsLink(env, orderId, order)}>
                            поискать их в YT
                        </a>
                    ) : (
                        <span>поискать их в YT</span>
                    )}
                </div>
            );
        }

        return (
            <>
                {pagination}

                <div className={b('table')}>
                    <div className={b('row', {head: true})}>
                        <div className={b('cell').mix(b('timestamp'))}>
                            timestamp
                        </div>
                        <div className={b('cell').mix(b('level'))}>level</div>
                        <div className={b('cell').mix(b('logger'))}>
                            logger + host
                        </div>
                        <div className={b('cell').mix(b('message'))}>
                            message
                        </div>
                    </div>

                    {logs.map(
                        (
                            {
                                level,
                                logger,
                                message,
                                timestamp,
                                host_name,
                                context,
                            },
                            index,
                        ) => (
                            <div key={index} className={b('row')}>
                                <div className={b('cell').mix(b('timestamp'))}>
                                    {getHumanMoscowFormatWithMilliseconds(
                                        timestamp,
                                    )}
                                </div>

                                <div className={b('cell').mix(b('level'))}>
                                    {level}
                                    {logger === USER_ACTION_LOGGER_NAME
                                        ? ' 🐾'
                                        : ''}
                                </div>

                                <div className={b('cell').mix(b('logger'))}>
                                    <span
                                        className={b('loggerText')}
                                        onClick={handleLogClick.bind(
                                            null,
                                            context,
                                        )}
                                    >
                                        {logger}
                                    </span>

                                    <br />

                                    {host_name}
                                </div>

                                <div
                                    className={b('cell').mix(
                                        b('message', {
                                            large:
                                                trimLarge &&
                                                message.length >
                                                    LARGE_MESSAGE_MINIMAL_LENGTH,
                                        }),
                                    )}
                                >
                                    {getJSONLikeString(message)}
                                </div>
                            </div>
                        ),
                    )}
                </div>

                {pagination}
            </>
        );
    }, [
        handleLogClick,
        pagination,
        isLoading,
        isFetched,
        error,
        logs,
        trimLarge,
        item_isFetched,
    ]);

    return (
        <div className={b()}>
            <Heading level="1">Логи {orderId}</Heading>

            <OrderLogsFilters />

            {content}

            <Modal
                isVisible={modalIsVisible}
                containerClassName={b('modalContent')}
                onClose={hideModal}
            >
                <Heading level="4">Контекст</Heading>

                {getJSONLikeString(modalContent)}
            </Modal>
        </div>
    );
};

export default OrderLogs;
