import React, {useCallback, useEffect, useState} from 'react';
import {useLocation, useHistory} from 'react-router-dom';
import {Select, TextInput} from 'lego-on-react';
import {block} from 'bem-cn';
import omit from 'lodash/omit';
import pick from 'lodash/pick';

import {DEFAULT_ORDER_LOGS_COUNT} from 'constants/search';
import {ORDER_LOGS_FILTER_PARAMS} from 'components/Order/OrderLogs/OrderLogsFilters/helpers/constants';

import {isOrderLogLevel, OrderLogLevel} from 'redux/reducers/types';

import parseOrderLogsUrl from 'lib/url/logs';
import getOrderLogsQueryParamsFromState from 'components/Order/OrderLogs/OrderLogsFilters/helpers/getOrderLogsQueryParamsFromState';
import {extendUrl} from 'lib/url/utils';
import getRecoveredOrderLogsStateFromQuery from 'components/Order/OrderLogs/OrderLogsFilters/helpers/getRecoveredOrdeLogsStateFromQuery';

import ActionButton from 'components/lego/ActionButton/ActionButton';

import './index.scss';

const b = block('OrderLogsFilters');

export interface IOrderLogsFilterFormState {
    level: OrderLogLevel;
    logger: string;
    message: string;
}

const OrderLogsFilters: React.FC = () => {
    const location = useLocation();

    const {search} = location;

    const history = useHistory();

    const [level, setLevel] = useState<OrderLogLevel>(OrderLogLevel.IRRELEVANT);
    const [logger, setLogger] = useState('');
    const [message, setMessage] = useState('');

    const recoverOrderLogsState = () => {
        const filterQueryParams = pick(
            parseOrderLogsUrl(search),
            ORDER_LOGS_FILTER_PARAMS,
        );

        const {level, logger, message} =
            getRecoveredOrderLogsStateFromQuery(filterQueryParams);

        setLevel(level);
        setLogger(logger);
        setMessage(message);
    };

    useEffect(() => {
        recoverOrderLogsState();
    }, []);

    const handleSubmit = useCallback(
        (e: React.FormEvent<HTMLFormElement>) => {
            e.preventDefault();

            const prevSearchQueryWithoutFilterParams = omit(
                parseOrderLogsUrl(search),
                ORDER_LOGS_FILTER_PARAMS,
            );

            const searchFilterQueryParams = getOrderLogsQueryParamsFromState({
                level,
                logger,
                message,
            });

            const newUrl = extendUrl(
                {
                    ...location,
                    search: '',
                },
                {
                    ...prevSearchQueryWithoutFilterParams,
                    ...searchFilterQueryParams,
                    page: '1',
                    count: `${DEFAULT_ORDER_LOGS_COUNT}`,
                    trimLarge: '1',
                },
            );

            if (newUrl !== `${location.pathname}${location.search}`) {
                history.push(newUrl);
            }
        },
        [location, search, level, logger, message, history],
    );

    const handleLevelChange = useCallback((newLevel: string | string[]) => {
        setLevel(
            isOrderLogLevel(newLevel[0])
                ? newLevel[0]
                : OrderLogLevel.IRRELEVANT,
        );
    }, []);

    const handleLoggerChange = useCallback((newLogger: string) => {
        setLogger(newLogger);
    }, []);

    const handleMessageChange = useCallback((newMessage: string) => {
        setMessage(newMessage);
    }, []);

    return (
        <form className={b()} onSubmit={handleSubmit}>
            <div className={b('formGroup')}>
                <div className={b('formGroupTitle')}>Статус</div>

                <Select
                    theme="normal"
                    size="s"
                    view="default"
                    tone="default"
                    type="radio"
                    width="max"
                    val={level}
                    items={Object.values(OrderLogLevel).map(value => ({
                        val: value,
                        text: value || 'Не важно',
                    }))}
                    onChange={handleLevelChange}
                />
            </div>

            <div className={b('formGroup')}>
                <div className={b('formGroupTitle')}>Имя лога</div>

                <TextInput
                    theme="normal"
                    size="m"
                    view="default"
                    tone="default"
                    text={logger}
                    hasClear
                    onChange={handleLoggerChange}
                />
            </div>

            <div className={b('formGroup')}>
                <div className={b('formGroupTitle')}>Часть текста</div>

                <TextInput
                    theme="normal"
                    size="m"
                    view="default"
                    tone="default"
                    text={message}
                    hasClear
                    onChange={handleMessageChange}
                />
            </div>

            <div className={b('formGroup', {bottom: true})}>
                <ActionButton className={b('button')} type="submit">
                    Найти
                </ActionButton>
            </div>
        </form>
    );
};

export default OrderLogsFilters;
