import React from 'react';
import {block} from 'bem-cn';

import {DEFAULT_ORDER_LOGS_COUNT} from 'constants/search';

import {IOrderInfo} from 'redux/reducers/types';
import {IHotelOrderItem} from 'redux/reducers/types/hotel';
import {ITrainOrderItem} from 'redux/reducers/types/train';

import getHumanMoscowFormat from 'lib/time/getHumanMoscowFormat';
import {getOrderStatusText} from 'lib/dictionaries/orderStatus';

import Link from 'components/basic/Link/Link';
import ClipboardButton from 'components/basic/ClipboardButton/ClipboardButton';
import HotelDatesOfState from 'components/Order/HotelDatesOfStates/HotelDatesOfState';

import './index.scss';

const b = block('OrderMainInfo');

interface IOwnProps {
    className?: string;
    isTechTexts: boolean;
    order: IOrderInfo;
    hotelOrderItem?: IHotelOrderItem;
    trainOrderItems?: ITrainOrderItem[];
}

const OrderMainInfo: React.FC<IOwnProps> = props => {
    const {className, isTechTexts, order, hotelOrderItem, trainOrderItems} =
        props;

    return (
        <div className={b.mix(className)}>
            <div>
                Дата и время оформления:{' '}
                {getHumanMoscowFormat(order.created_at)}
            </div>
            <div className={b('row')}>
                UID заказа в Путешествиях: {order.id}
                <ClipboardButton text={order.id} size="s" />
            </div>

            <div>
                Статус заказа: {getOrderStatusText(order.status, isTechTexts)}
            </div>

            {hotelOrderItem && (
                <HotelDatesOfState
                    checkinDate={hotelOrderItem.request_info.checkin_date}
                    checkoutDate={hotelOrderItem.request_info.checkout_date}
                    orderId={order.id}
                    adminActionToken={order.admin_action_token}
                    canModifyHotelOrderDetails={
                        order.flags.can_modify_hotel_order_details
                    }
                />
            )}

            <div>Действия клиента до оформления заказа: нет информации</div>

            <a href={order.url} target="_blank" rel="noopener noreferrer">
                Заказ в портале Путешествий
            </a>

            {trainOrderItems &&
                trainOrderItems.map((trainOrderItem, index) => (
                    <>
                        <div>
                            Номер РЖД заказа
                            {trainOrderItems.length > 1 ? ` #${index + 1}` : ''}
                            :{` ${trainOrderItem.partner_number} `}
                            <a
                                href={`/api/orders/${order.id}/downloadBlank/${trainOrderItem.passengers_info[0]?.ticket?.blank_id}`}
                                download
                            >
                                маршрут-квитанция
                            </a>
                        </div>
                    </>
                ))}

            <div>
                <Link to={`/${order.id}/workflow/${order.workflow_id}`}>
                    Воркфлоу
                </Link>
            </div>

            <div>
                <Link to={`/${order.id}/payloads`}>Информация об услугах</Link>
            </div>

            <div>
                <Link
                    to={`/${order.id}/logs?page=1&count=${DEFAULT_ORDER_LOGS_COUNT}&trimLarge=1`}
                >
                    Логи
                </Link>
            </div>
        </div>
    );
};

export default OrderMainInfo;
