import React, {useCallback, useMemo, useState} from 'react';
import {useDispatch, useSelector} from 'react-redux';
import {RadioBox, Select} from 'lego-on-react';
import {Loader} from '@yandex-data-ui/common';
import {useToasts} from 'react-toast-notifications';
import {block} from 'bem-cn';
import {orderSendNotificationRequest} from 'redux/reducers/order/actions';

import {
    DisplayOrderType,
    IOrderInfo,
    isOrderNotificationType,
    IStore,
    OrderNotificationTransport,
    OrderNotificationType,
    OrderRefundType,
} from 'redux/reducers/types';

import getHumanMoscowFormatWithMilliseconds from 'lib/time/getHumanMoscowFormatWithMilliseconds';

import Heading from 'components/basic/Heading/Heading';
import ActionButton from 'components/lego/ActionButton/ActionButton';

import './index.scss';

const b = block('OrderNotifications');

interface ISelectItem {
    val: string;
    text: string;
}

interface IOrderNotificationsProps {
    className?: string;
    order: IOrderInfo;
}

function isUserRefund(type: OrderRefundType): boolean {
    return (
        type === OrderRefundType.RT_TRAIN_USER_REFUND ||
        type === OrderRefundType.RT_GENERIC_USER_REFUND
    );
}

const OrderNotifications: React.FC<IOrderNotificationsProps> = props => {
    const {
        className,
        order: {
            id: orderId,
            display_order_type,
            order_refunds,
            payment_info,
            flags: {can_resend_successful_mail, can_resend_refund_mail},
        },
    } = props;

    const {
        order: {
            sendingNotification: {isLoading},
        },
    } = useSelector((store: IStore) => store);

    const [notificationType, setNotificationType] =
        useState<OrderNotificationType>(
            can_resend_successful_mail
                ? OrderNotificationType.SUCCESS
                : OrderNotificationType.REFUND,
        );

    const selectItems = useMemo((): ISelectItem[] | null => {
        if (
            !order_refunds ||
            display_order_type !== DisplayOrderType.DT_TRAIN
        ) {
            return null;
        }

        const items: ISelectItem[] = [];

        order_refunds.forEach(({id, created_at, type, refunded_amount}) => {
            if (isUserRefund(type)) {
                const created =
                    getHumanMoscowFormatWithMilliseconds(created_at);

                if (!refunded_amount) {
                    items.push({
                        val: id,
                        text: `${created}`,
                    });
                } else {
                    const {value, currency} = refunded_amount;

                    items.push({
                        val: id,
                        text: `${created} amount: ${value} ${currency}`,
                    });
                }
            }
        });

        return items.length ? items : null;
    }, [display_order_type, order_refunds, payment_info]);

    const [refundId, setRefundId] = useState<string | null>(
        selectItems?.[0]?.val || null,
    );

    const dispatch = useDispatch();

    const {addToast} = useToasts();

    const handleChange = useCallback(
        (e: React.ChangeEvent<HTMLInputElement>) => {
            const notificationTypeFromInput = e.target.value;

            if (isOrderNotificationType(notificationTypeFromInput)) {
                setNotificationType(notificationTypeFromInput);
            }
        },
        [],
    );

    const handleClick = useCallback(() => {
        dispatch(
            orderSendNotificationRequest({
                orderId,
                transport: OrderNotificationTransport.EMAIL,
                orderRefundId: refundId || undefined,
                type: notificationType,
                addToast,
            }),
        );
    }, [orderId, notificationType, refundId, addToast, dispatch]);

    const handleSelectChange = useCallback((newRefundId: string | string[]) => {
        setRefundId(newRefundId[0]);
    }, []);

    if (!can_resend_successful_mail && !can_resend_refund_mail) {
        return null;
    }

    return (
        <div className={b.mix(className)}>
            <Heading level="4">Повторные уведомления</Heading>

            <RadioBox
                theme="normal"
                size="m"
                view="default"
                tone="default"
                value={notificationType}
                name="default"
                onChange={handleChange}
            >
                <RadioBox.Radio
                    value={OrderNotificationType.SUCCESS}
                    disabled={!can_resend_successful_mail}
                >
                    О покупке
                </RadioBox.Radio>

                <RadioBox.Radio
                    value={OrderNotificationType.REFUND}
                    disabled={
                        !can_resend_refund_mail ||
                        (display_order_type === DisplayOrderType.DT_TRAIN &&
                            !selectItems)
                    }
                >
                    <div className={b('refundWrapper')}>
                        <span>О возврате</span>

                        {display_order_type === DisplayOrderType.DT_TRAIN &&
                            selectItems &&
                            refundId && (
                                <Select
                                    cls={b('refundIdSelect')}
                                    theme="normal"
                                    size="s"
                                    view="default"
                                    tone="default"
                                    type="radio"
                                    val={refundId}
                                    disabled={
                                        notificationType !==
                                        OrderNotificationType.REFUND
                                    }
                                    items={selectItems}
                                    onChange={handleSelectChange}
                                />
                            )}
                    </div>
                </RadioBox.Radio>
            </RadioBox>

            {isLoading ? (
                <Loader size="m" className={b('loader').toString()} />
            ) : (
                <ActionButton
                    className={b('button').toString()}
                    type="submit"
                    onClick={handleClick}
                >
                    Отправить уведомление
                </ActionButton>
            )}
        </div>
    );
};

export default OrderNotifications;
