import React, {useMemo} from 'react';
import moment, {Moment} from 'moment';
import {block} from 'bem-cn';

import {ITrainOrderItem} from 'redux/reducers/types/train';

import {TRAIN_ORDER_DOCUMENT_TYPE_DICT} from 'lib/dictionaries/trainOrderDocumentType';
import {TRAIN_ORDER_PASSENGER_CATEGORY_DICT} from 'lib/dictionaries/trainOrderPassengerCategory';
import {TRAIN_ORDER_PASSENGER_TICKET_BLANK_STATUS_DICT} from 'lib/dictionaries/trainOrderPassengerTicketBlankStatus';
import getPersonalFieldText from 'lib/getPersonalFieldText';

import OrderUpdateTrainTickets from 'components/Order/OrderPassengers/OrderUpdateTrainTickets/OrderUpdateTrainTickets';
import Heading from 'components/basic/Heading/Heading';

import './index.scss';

const b = block('OrderPassengers');

interface IOrderPassengersProps {
    className?: string;
    trainOrderItem: ITrainOrderItem;
    hasMaskedInfo: boolean;
}

const OrderPassengers: React.FC<IOrderPassengersProps> = props => {
    const {
        className,
        trainOrderItem: {passengers_info, car_number},
        hasMaskedInfo,
    } = props;

    const now = useMemo((): Moment => moment(), []);

    return (
        <div className={b.mix(className)}>
            <Heading level="4">Пассажиры</Heading>

            <div className={b('passengers')}>
                {passengers_info.map(
                    (
                        {
                            first_name,
                            last_name,
                            patronymic,
                            birthday,
                            document_type,
                            document_number,
                            category,
                            insurance,
                            ticket,
                        },
                        index,
                    ) => {
                        if (!ticket) {
                            return (
                                <div className={b('passenger')} key={index}>
                                    Нет билета
                                </div>
                            );
                        }

                        const {
                            places,
                            im_blank_status,
                            total_cost,
                            tariff_amount,
                            service_amount,
                            fee_amount,
                        } = ticket;

                        const insuranceAmount =
                            insurance?.partner_operation_status
                                ? insurance.amount.value
                                : 0;

                        return (
                            <div className={b('passenger')} key={index}>
                                {getPersonalFieldText(
                                    first_name,
                                    hasMaskedInfo,
                                )}{' '}
                                {getPersonalFieldText(last_name, hasMaskedInfo)}{' '}
                                {getPersonalFieldText(
                                    patronymic,
                                    hasMaskedInfo,
                                )}
                                <br />
                                👤{' '}
                                {getPersonalFieldText(
                                    birthday ? now.diff(birthday, 'year') : '',
                                    hasMaskedInfo,
                                )}{' '}
                                {TRAIN_ORDER_DOCUMENT_TYPE_DICT[document_type]}{' '}
                                {getPersonalFieldText(
                                    document_number,
                                    hasMaskedInfo,
                                )}
                                <br />
                                {places
                                    .map(
                                        ({number}) =>
                                            `🚃 ${car_number} 💺 ${number}`,
                                    )
                                    .join(', ')}{' '}
                                {TRAIN_ORDER_PASSENGER_CATEGORY_DICT[category]}
                                <br />
                                Статус РЖД:{' '}
                                {im_blank_status
                                    ? TRAIN_ORDER_PASSENGER_TICKET_BLANK_STATUS_DICT[
                                          im_blank_status
                                      ]
                                    : 'нет'}
                                <br />
                                {(
                                    tariff_amount.value + service_amount.value
                                ).toFixed(2)}{' '}
                                +{' '}
                                {insuranceAmount ? `${insuranceAmount} + ` : ''}
                                {fee_amount.value} ={' '}
                                {total_cost.value + insuranceAmount}
                            </div>
                        );
                    },
                )}
            </div>

            <OrderUpdateTrainTickets className={b('updateTrainTickets')} />
        </div>
    );
};

export default OrderPassengers;
