import React, {useCallback, useState} from 'react';
import {useDispatch, useSelector} from 'react-redux';
import {compose} from '@bem-react/core';
import {Loader} from '@yandex-data-ui/common';
import {useToasts} from 'react-toast-notifications';
import {block} from 'bem-cn';
import {orderEditPayloadRequest} from 'redux/reducers/order/actions';

import {IOrderPayloadInfo, IStore} from 'redux/reducers/types';

import {
    TabsMenu as LegoTabsMenu,
    withSizeM,
    withLayoutHoriz,
    withViewDefault,
} from '@yandex-lego/components/TabsMenu/desktop';
import {TabsPanes} from '@yandex-lego/components/TabsPanes/desktop/bundle';
import OrderPayloadEditor, {
    IEditorType,
} from 'components/Order/OrderPayloads/OrderPayload/OrderPayloadEditor/OrderPayloadEditor';
import Heading from 'components/basic/Heading/Heading';
import Modal from 'components/basic/Modal/Modal';
import ActionButton from 'components/lego/ActionButton/ActionButton';
import OrderPayloadDiffViewer from 'components/Order/OrderPayloads/OrderPayload/OrderPayloadDiffViewer/OrderPayloadDiffViewer';

import './index.scss';

const b = block('OrderPayload');
const TabsMenu = compose(
    withSizeM,
    withViewDefault,
    withLayoutHoriz,
)(LegoTabsMenu);

interface IOrderPayloadProps {
    orderId: string;
    payloadInfo: IOrderPayloadInfo;
}

const OrderPayload: React.FC<IOrderPayloadProps> = props => {
    const {orderId, payloadInfo} = props;

    const {
        order: {
            payloads: {
                edit: {isLoading},
            },
        },
    } = useSelector((store: IStore) => store);

    const dispatch = useDispatch();

    const {addToast} = useToasts();

    const [activeTab, setActiveTab] = useState('editor');
    const [oldPayload, setOldPayload] = useState(payloadInfo.payload);
    const [currentPayload, setCurrentPayload] = useState(payloadInfo.payload);
    const [hasChanges, setHasChanges] = useState(false);
    const [isModalOpen, setIsModalOpen] = useState(false);

    const handleChange = useCallback(
        editedPayload => {
            setCurrentPayload(editedPayload);

            if (!hasChanges) {
                setHasChanges(true);
            }
        },
        [hasChanges],
    );

    const handleSave = useCallback(() => {
        if (!hasChanges) {
            return;
        }

        setIsModalOpen(true);
    }, [hasChanges]);

    const handleConfirmSave = useCallback(() => {
        dispatch(
            orderEditPayloadRequest({
                order_id: orderId,
                order_item_id: payloadInfo.order_item_id,
                payload: currentPayload,
                addToast,
            }),
        );

        setHasChanges(false);
        setIsModalOpen(false);
        setOldPayload(currentPayload);
    }, [orderId, payloadInfo.order_item_id, addToast, currentPayload]);

    const handleModalClose = useCallback(() => setIsModalOpen(false), []);

    const getEditor = (payload: object, editorType: IEditorType) => (
        <OrderPayloadEditor
            orderId={payloadInfo.order_item_id}
            payload={payload}
            type={editorType}
            onChange={handleChange}
            onSave={handleSave}
        />
    );

    return (
        <div className={b()}>
            <TabsMenu
                size="m"
                view="default"
                layout="horiz"
                activeTab={activeTab}
                tabs={[
                    {
                        id: 'editor',
                        onClick: () => setActiveTab('editor'),
                        content: 'Старый редактор',
                    },
                    {
                        id: 'ace-editor',
                        onClick: () => setActiveTab('ace-editor'),
                        content: 'Новый редактор',
                    },
                    {
                        id: 'textarea',
                        onClick: () => setActiveTab('textarea'),
                        content: 'Textarea',
                    },
                ]}
            />

            <TabsPanes
                activePane={activeTab}
                panes={[
                    {
                        id: 'editor',
                        content: getEditor(currentPayload, 'editor'),
                    },
                    {
                        id: 'ace-editor',
                        content: getEditor(currentPayload, 'ace-editor'),
                    },
                    {
                        id: 'textarea',
                        content: getEditor(currentPayload, 'textarea'),
                    },
                ]}
            />

            {isLoading ? (
                <Loader className={b('loader').toString()} />
            ) : (
                <ActionButton
                    className={b('saveButton').toString()}
                    type="submit"
                    disabled={Boolean(!hasChanges)}
                    onClick={handleSave}
                >
                    Сохранить
                </ActionButton>
            )}

            <Modal
                isVisible={isModalOpen}
                containerClassName={b('modalContent')}
                onClose={handleModalClose}
            >
                <Heading level="4">Сохранить изменения?</Heading>

                <OrderPayloadDiffViewer
                    oldValue={JSON.stringify(oldPayload, null, 4)}
                    newValue={JSON.stringify(currentPayload, null, 4)}
                />

                <ActionButton
                    className={b('refundApprovalButton').toString()}
                    onClick={handleConfirmSave}
                >
                    Сохранить
                </ActionButton>
            </Modal>
        </div>
    );
};

export default OrderPayload;
