import React, {useMemo, Fragment} from 'react';
import {block} from 'bem-cn';

import {
    ETrustRefundType,
    IOrderPaymentInfoItem,
    IStoreSettings,
    OrderPaymentInfoReceiptTypes,
    OrderPaymentStatus,
} from 'redux/reducers/types';
import {Environment} from 'shared/types';

import getHumanMoscowFormat from 'lib/time/getHumanMoscowFormat';
import {getOrderPaymentStatusText} from 'lib/dictionaries/orderPaymentStatus';
import {getOrderRefundStatusText} from 'lib/dictionaries/orederRefundStatus';
import {getOrderResizeStatusText} from 'lib/dictionaries/orederResizeStatus';

import Heading from 'components/basic/Heading/Heading';

import './index.scss';

const b = block('OrderPaymentsAndRefunds');

interface IOrderPaymentsAndRefundsProps {
    className?: string;
    paymentInfo: IOrderPaymentInfoItem[];
    settings: IStoreSettings;
    env: Environment;
}

function getPaymentTotal(payment: IOrderPaymentInfoItem): number {
    if (payment.status === OrderPaymentStatus.IS_PAYMENT_NOT_AUTHORIZED) {
        // shouldn't be accounted in the total sum
        return 0;
    }

    return payment.current_price.value;
}

const OrderPaymentsAndRefunds: React.FC<
    IOrderPaymentsAndRefundsProps
> = props => {
    const {className, paymentInfo, settings, env} = props;

    const refundsAmounts = useMemo(() => {
        return paymentInfo.map(
            ({trust_refunds}) =>
                trust_refunds?.map(({refund_items, id}) => ({
                    amount: refund_items.reduce(
                        (accSum, {original_amount, target_amount}) =>
                            accSum +
                            original_amount.value -
                            target_amount.value,
                        0,
                    ),
                    id,
                })) || [],
        );
    }, [paymentInfo]);

    if (!paymentInfo.length) {
        return null;
    }

    let needToShowRestAmount = false;

    return (
        <div className={b.mix(className)}>
            <Heading level="4">Платежи и возвраты</Heading>

            <div className={b('table')}>
                <div className={b('row', {head: true})}>
                    <div className={b('cell')}>Дата и время</div>
                    <div className={b('cell')}>Тип</div>
                    <div className={b('cell')}>Идентификатор</div>
                    <div className={b('cell')}>Статус операции</div>
                    <div className={b('cell')}>Код ошибки</div>
                    <div className={b('cell')}>Расшифровка</div>
                    <div className={b('cell')}>RRN</div>
                    <div className={b('cell')}>Код авторизации</div>
                    <div className={b('cell')}>Маска карты</div>
                    <div className={b('cell')}>Сумма</div>
                    <div className={b('cell')}>Чек</div>
                </div>

                {paymentInfo.map((paymentInfoItem, index) => {
                    const {
                        paid_at,
                        trust_refunds,
                        trust_payment_id,
                        status,
                        payment_resp_code,
                        payment_resp_desc,
                        rrn,
                        approval_code,
                        card_number,
                        receipts,
                        original_price,
                    } = paymentInfoItem;

                    const paymentReceipt = receipts.find(
                        ({type}) =>
                            type === OrderPaymentInfoReceiptTypes.FRT_ACQUIRE,
                    );
                    const paymentClearReceipt = receipts.find(
                        ({type}) =>
                            type === OrderPaymentInfoReceiptTypes.FRT_CLEAR,
                    );

                    if (paymentClearReceipt || trust_refunds) {
                        needToShowRestAmount = true;
                    }

                    return (
                        <Fragment key={index}>
                            <div className={b('row')}>
                                <div className={b('cell')}>
                                    {paid_at
                                        ? getHumanMoscowFormat(paid_at)
                                        : ''}
                                </div>

                                <div className={b('cell')}>
                                    {status ===
                                    OrderPaymentStatus.IS_PAYMENT_NOT_AUTHORIZED
                                        ? 'Попытка платежа'
                                        : 'Платеж'}
                                </div>

                                <div className={b('cell')}>
                                    {trust_payment_id ? (
                                        <a
                                            href={`https://scrooge${
                                                env === Environment.production
                                                    ? ''
                                                    : '-test'
                                            }.paysys.yandex-team.ru/payments/?trust_payment_id=${trust_payment_id}`}
                                            target="_blank"
                                            rel="noopener noreferrer"
                                        >
                                            {trust_payment_id}
                                        </a>
                                    ) : (
                                        ''
                                    )}
                                </div>

                                <div className={b('cell')}>
                                    {getOrderPaymentStatusText(
                                        status,
                                        settings.isTechTexts,
                                    )}
                                </div>
                                <div className={b('cell')}>
                                    {payment_resp_code || ''}
                                </div>
                                <div className={b('cell')}>
                                    {payment_resp_desc || ''}
                                </div>
                                <div className={b('cell')}>{rrn || ''}</div>
                                <div className={b('cell')}>
                                    {approval_code || ''}
                                </div>
                                <div className={b('cell')}>
                                    {card_number || ''}
                                </div>
                                <div className={b('cell')}>
                                    {original_price.value.toFixed(2)}
                                </div>

                                <div className={b('cell')}>
                                    {paymentReceipt ? (
                                        <a
                                            href={paymentReceipt.url}
                                            target="_blank"
                                            rel="noopener noreferrer"
                                        >
                                            Чек
                                        </a>
                                    ) : (
                                        ''
                                    )}
                                </div>
                            </div>

                            {trust_refunds
                                ?.sort(
                                    (
                                        {created_at: createdAt1},
                                        {created_at: createdAt2},
                                    ) =>
                                        Number(new Date(createdAt1)) -
                                        Number(new Date(createdAt2)),
                                )
                                .map(refund => {
                                    const refundReceipt = receipts.find(
                                        ({url}) =>
                                            url.indexOf(
                                                refund.trust_refund_id,
                                            ) !== -1,
                                    );
                                    const refundAmount = refundsAmounts[
                                        index
                                    ].find(({id}) => id === refund.id);

                                    return (
                                        <div
                                            key={refund.id}
                                            className={b('row')}
                                        >
                                            <div className={b('cell')}>
                                                {refund.created_at
                                                    ? getHumanMoscowFormat(
                                                          refund.created_at,
                                                      )
                                                    : ''}
                                            </div>

                                            <div className={b('cell')}>
                                                {refund.type ===
                                                ETrustRefundType.REFUND
                                                    ? 'Возврат'
                                                    : 'Возврат с отложенным клирингом'}
                                            </div>

                                            <div className={b('cell')}>
                                                {refund.trust_refund_id && (
                                                    <a
                                                        href={`https://scrooge${
                                                            env ===
                                                            Environment.production
                                                                ? ''
                                                                : '-test'
                                                        }.paysys.yandex-team.ru/refunds/?trust_refund_id=${
                                                            refund.trust_refund_id
                                                        }`}
                                                        target="_blank"
                                                        rel="noopener noreferrer"
                                                    >
                                                        {refund.trust_refund_id}
                                                    </a>
                                                )}
                                            </div>

                                            <div className={b('cell')}>
                                                {refund.type ===
                                                ETrustRefundType.REFUND
                                                    ? getOrderRefundStatusText(
                                                          refund.state,
                                                          settings.isTechTexts,
                                                      )
                                                    : getOrderResizeStatusText(
                                                          refund.state,
                                                          settings.isTechTexts,
                                                      )}
                                            </div>
                                            <div className={b('cell')} />
                                            <div className={b('cell')} />
                                            <div className={b('cell')} />
                                            <div className={b('cell')} />
                                            <div className={b('cell')} />
                                            <div className={b('cell')}>
                                                {refundAmount &&
                                                    refundAmount.amount.toFixed(
                                                        2,
                                                    )}
                                            </div>

                                            <div className={b('cell')}>
                                                {refundReceipt &&
                                                    (refund.type ===
                                                        ETrustRefundType.REFUND ||
                                                        paymentClearReceipt) && (
                                                        <a
                                                            href={
                                                                refund.type ===
                                                                ETrustRefundType.REFUND
                                                                    ? refundReceipt.url
                                                                    : paymentClearReceipt?.url
                                                            }
                                                            target="_blank"
                                                            rel="noopener noreferrer"
                                                        >
                                                            Чек
                                                        </a>
                                                    )}
                                            </div>
                                        </div>
                                    );
                                })}
                        </Fragment>
                    );
                })}

                {needToShowRestAmount && (
                    <div className={b('row')}>
                        <div className={b('cell')} />
                        <div className={b('cell')} />
                        <div className={b('cell')} />
                        <div className={b('cell')} />
                        <div className={b('cell')} />
                        <div className={b('cell')} />
                        <div className={b('cell')} />
                        <div className={b('cell')} />
                        <div className={b('cell')}>Итого остаток: </div>
                        <div className={b('cell')}>
                            {paymentInfo
                                .reduce(
                                    (accCurrentPrice, payment) =>
                                        accCurrentPrice +
                                        getPaymentTotal(payment),
                                    0,
                                )
                                .toFixed(2)}
                        </div>
                        <div className={b('cell')} />
                    </div>
                )}
            </div>
        </div>
    );
};

export default OrderPaymentsAndRefunds;
