import React, {useCallback, useState} from 'react';
import {useDispatch} from 'react-redux';
import {RadioBox, Select, TextArea} from 'lego-on-react';
import {Loader} from '@yandex-data-ui/common';
import {block} from 'bem-cn';
import {useToasts} from 'react-toast-notifications';
import {orderRefundRequest} from 'redux/reducers/order/actions';

import {IOrderFlags, IOrderInfo} from 'redux/reducers/types';

import Heading from 'components/basic/Heading/Heading';
import Modal from 'components/basic/Modal/Modal';
import MoneyInput from 'components/basic/MoneyInput/MoneyInput';
import ActionButton from 'components/lego/ActionButton/ActionButton';

import './index.scss';

const b = block('OrderRefund');

interface IOrderRefundProps {
    className?: string;
    order: IOrderInfo;
    isLoading: boolean;
}

export enum OrderRefundType {
    ALL = 'ALL',
    FEE = 'FEE',
    PARTLY = 'PARTLY',
}

const ORDER_REFUND_TYPES = Object.values(OrderRefundType);

function isOrderRefundType(refundType: any): refundType is OrderRefundType {
    return ORDER_REFUND_TYPES.includes(refundType);
}

function getDefaultRefundType(flags: IOrderFlags): OrderRefundType {
    if (flags.can_manual_refund_all_money) {
        return OrderRefundType.ALL;
    }

    if (flags.can_manual_refund_yandex_fee) {
        return OrderRefundType.FEE;
    }

    return OrderRefundType.PARTLY;
}

const OrderRefund: React.FC<IOrderRefundProps> = props => {
    const {
        className,
        order: {
            id: orderId,
            admin_action_token,
            payment_info,
            flags,
            flags: {
                can_manual_refund_all_money,
                can_manual_refund_yandex_fee,
                can_manual_refund_by_fiscal_item,
            },
        },
        isLoading,
    } = props;

    const firstPaymentInfo = payment_info[0] as
        | typeof payment_info[0]
        | undefined;

    const invoice_items = firstPaymentInfo?.invoice_items;

    const dispatch = useDispatch();

    const {addToast} = useToasts();

    const [refundType, setRefundType] = useState<OrderRefundType>(
        getDefaultRefundType(flags),
    );
    const [fiscalItemId, setFiscalItemId] = useState<number>(
        invoice_items?.[0].fiscal_item_id || 0,
    );
    const [partlyRefundAmount, setPartlyRefundAmount] = useState<number>(
        (invoice_items && invoice_items?.[0].money_amount.value) || 0,
    );
    const [reason, setReason] = useState<string>('');
    const [isModalOpen, setIsModalOpen] = useState<boolean>(false);
    const [isRefundButtonDisabled, setRefundButtonDisabled] =
        useState<boolean>(false);

    const handleChange = useCallback(
        (e: React.ChangeEvent<HTMLInputElement>) => {
            const currentRefundType = e.target.value;

            if (isOrderRefundType(currentRefundType)) {
                setRefundType(currentRefundType);
            }
        },
        [],
    );

    const handleSelectChange = useCallback(
        (newFiscalItemId: string | string[]) => {
            if (!invoice_items) {
                return;
            }

            const fiscalItemIdNumber = Number(newFiscalItemId[0]);

            setFiscalItemId(fiscalItemIdNumber);

            const invoiceItem = invoice_items.find(
                ({fiscal_item_id}) => fiscal_item_id === fiscalItemIdNumber,
            );

            if (invoiceItem) {
                setPartlyRefundAmount(invoiceItem.money_amount.value);
            }
        },
        [invoice_items],
    );

    const handlePartlyAmountChange = useCallback(
        (isValid: boolean, value: number) => {
            if (isValid) {
                setPartlyRefundAmount(value);
            }

            setRefundButtonDisabled(!isValid);
        },
        [],
    );

    const handleRefundClick = useCallback(
        () => setIsModalOpen(!isModalOpen),
        [isModalOpen],
    );

    const handleModalClose = useCallback(() => setIsModalOpen(false), []);

    const handleReasonChange = useCallback((newReason: string) => {
        if (newReason.length > 500) {
            newReason = newReason.substr(0, 500);
        }
        setReason(newReason);
    }, []);

    const handleApproveButtonClick = useCallback(() => {
        const commonParams = {
            orderId,
            addToast,
            adminActionToken: admin_action_token,
            reason,
        };

        const orderRefundRequestParams =
            refundType === OrderRefundType.PARTLY
                ? {
                      ...commonParams,
                      refundType,
                      refundsByFiscalItem: {
                          [fiscalItemId]: {
                              value: partlyRefundAmount,
                              currency: 'RUB',
                          },
                      },
                  }
                : {
                      ...commonParams,
                      refundType,
                      refundsByFiscalItem: null,
                  };

        dispatch(orderRefundRequest(orderRefundRequestParams));

        setIsModalOpen(false);
        setReason('');
    }, [
        orderId,
        addToast,
        admin_action_token,
        reason,
        refundType,
        fiscalItemId,
        partlyRefundAmount,
        dispatch,
    ]);

    if (
        (!can_manual_refund_all_money &&
            !can_manual_refund_yandex_fee &&
            !can_manual_refund_by_fiscal_item) ||
        !invoice_items
    ) {
        return null;
    }

    return (
        <div className={b.mix(className)}>
            <Heading level="4">Возврат</Heading>

            <div>
                <RadioBox
                    theme="normal"
                    size="m"
                    view="default"
                    tone="default"
                    value={refundType}
                    name="default"
                    onChange={handleChange}
                >
                    <RadioBox.Radio
                        value={OrderRefundType.ALL}
                        disabled={!can_manual_refund_all_money}
                    >
                        Полная сумма по заказу
                    </RadioBox.Radio>

                    <RadioBox.Radio
                        value={OrderRefundType.FEE}
                        disabled={!can_manual_refund_yandex_fee}
                    >
                        Только сервисный сбор Яндекса
                    </RadioBox.Radio>

                    <RadioBox.Radio
                        value={OrderRefundType.PARTLY}
                        disabled={!can_manual_refund_by_fiscal_item}
                    >
                        <div className={b('partlyRefundWrapper')}>
                            Частичный возврат
                            <Select
                                cls={b('fiscalIdSelect')}
                                theme="normal"
                                size="s"
                                view="default"
                                tone="default"
                                type="radio"
                                val={fiscalItemId}
                                disabled={refundType !== OrderRefundType.PARTLY}
                                items={invoice_items.map(
                                    ({
                                        fiscal_item_id,
                                        fiscal_item_type,
                                        money_amount,
                                    }) => ({
                                        val: fiscal_item_id,
                                        text: `${fiscal_item_type} - ${money_amount.value}${money_amount.currency}`,
                                    }),
                                )}
                                onChange={handleSelectChange}
                            />
                            <MoneyInput
                                value={partlyRefundAmount}
                                disabled={refundType !== OrderRefundType.PARTLY}
                                onChange={handlePartlyAmountChange}
                            />
                        </div>
                    </RadioBox.Radio>
                </RadioBox>
            </div>

            {isLoading ? (
                <Loader size="m" className={b('loader').toString()} />
            ) : (
                <ActionButton
                    className={b('button').toString()}
                    type="submit"
                    onClick={handleRefundClick}
                    disabled={isRefundButtonDisabled}
                >
                    Сделать возврат
                </ActionButton>
            )}

            <Modal
                isVisible={isModalOpen}
                containerClassName={b('modalContent')}
                onClose={handleModalClose}
            >
                <Heading level="4">Причина возврата</Heading>

                <TextArea
                    theme="normal"
                    rows={3}
                    size="m"
                    text={reason}
                    onChange={handleReasonChange}
                    hasClear
                />

                <div className={b('partlyRefundAmount')}>
                    Сумма возврата: {partlyRefundAmount}
                </div>

                <ActionButton
                    className={b('refundApprovalButton').toString()}
                    disabled={!reason}
                    onClick={handleApproveButtonClick}
                >
                    Подтверждаю, сделать возврат
                </ActionButton>
            </Modal>
        </div>
    );
};

export default OrderRefund;
