import React, {useCallback, useMemo, useState} from 'react';
import {useDispatch, useSelector} from 'react-redux';
import {Select} from 'lego-on-react';
import {Loader} from '@yandex-data-ui/common';
import {useToasts} from 'react-toast-notifications';
import {block} from 'bem-cn';
import {orderRetryMoneyRefundRequest} from 'redux/reducers/order/actions';

import {IOrderInfo, IStore} from 'redux/reducers/types';

import getHumanMoscowFormatWithMilliseconds from 'lib/time/getHumanMoscowFormatWithMilliseconds';

import Heading from 'components/basic/Heading/Heading';
import ActionButton from 'components/lego/ActionButton/ActionButton';

import './index.scss';

const b = block('OrderRetryMoneyRefund');

interface ISelectItem {
    val: string;
    text: string;
}

interface IOrderRetryMoneyRefundProps {
    className?: string;
    order: IOrderInfo;
}

const OrderRetryMoneyRefund: React.FC<IOrderRetryMoneyRefundProps> = props => {
    const {
        className,
        order: {
            id: orderId,
            admin_action_token: adminActionToken,
            display_order_type,
            order_refunds,
            payment_info,
            flags: {can_retry_money_refund},
        },
    } = props;

    const {
        order: {
            retryingMoneyRefund: {isLoading},
        },
    } = useSelector((store: IStore) => store);

    const selectItems = useMemo((): ISelectItem[] | null => {
        if (!order_refunds) {
            return null;
        }

        const items: ISelectItem[] = [];

        order_refunds.forEach(({id, created_at, refunded_amount}) => {
            const created = getHumanMoscowFormatWithMilliseconds(created_at);

            if (!refunded_amount) {
                items.push({
                    val: id,
                    text: `${created}`,
                });
            } else {
                const {value, currency} = refunded_amount;

                items.push({
                    val: id,
                    text: `${created} amount: ${value} ${currency}`,
                });
            }
        });

        return items.length ? items : null;
    }, [display_order_type, order_refunds, payment_info]);

    const [refundId, setRefundId] = useState<string | null>(
        selectItems?.[0]?.val || null,
    );

    const dispatch = useDispatch();

    const {addToast} = useToasts();

    const handleClick = useCallback(() => {
        dispatch(
            orderRetryMoneyRefundRequest({
                orderId,
                adminActionToken,
                orderRefundId: refundId || undefined,
                addToast,
            }),
        );
    }, [orderId, adminActionToken, refundId, addToast, dispatch]);

    const handleSelectChange = useCallback((newRefundId: string | string[]) => {
        setRefundId(newRefundId[0]);
    }, []);

    if (!can_retry_money_refund) {
        return null;
    }

    return (
        <div className={b.mix(className)}>
            <Heading level="4">Повторная попытка возврата средств</Heading>

            {selectItems && refundId && (
                <Select
                    cls={b('refundIdSelect')}
                    theme="normal"
                    size="s"
                    view="default"
                    tone="default"
                    type="radio"
                    val={refundId}
                    items={selectItems}
                    onChange={handleSelectChange}
                />
            )}

            {isLoading ? (
                <Loader size="m" className={b('loader').toString()} />
            ) : (
                <ActionButton
                    className={b('button').toString()}
                    type="submit"
                    onClick={handleClick}
                >
                    Повторить возврат
                </ActionButton>
            )}
        </div>
    );
};

export default OrderRetryMoneyRefund;
