import React from 'react';
import ReactDOM from 'react-dom';
import {block} from 'bem-cn';
import moment from 'moment-timezone';

import {EMirEligibility, IMirPromo, IOrderInfo} from 'redux/reducers/types';

import {getOrderStatusText} from 'lib/dictionaries/orderStatus';

import OrderShortInfoIcon from 'components/Order/OrderShotInfo/OrderShotInfoIcon/OrderShotInfoIcon';

import './index.scss';

const b = block('OrderShortInfo');

interface IOrderShotInfoProps {
    order: IOrderInfo | null;
    isTechTexts: boolean;
}

function isMir(mirPromo?: IMirPromo | null): boolean {
    if (!mirPromo) {
        return false;
    }

    const {offer_eligibility, paid_with_mir} = mirPromo;

    return (
        offer_eligibility === EMirEligibility.MIR_ELIGIBILE &&
        Boolean(paid_with_mir)
    );
}

const OrderShortInfo: React.FC<IOrderShotInfoProps> = props => {
    const {order, isTechTexts} = props;

    if (order === null) {
        return null;
    }

    if (order.hotel_order_items.length <= 0) {
        return null;
    }

    if (typeof document === 'undefined') {
        return null;
    }

    const hotelOrderItem = order.hotel_order_items[0];
    const status = order.status;
    const hasPromoCode =
        order.order_price_info.promo_code_application_results.length > 0;
    const hasDiscount = order.order_price_info.discount_amount?.value > 0;
    const isFuture =
        moment(hotelOrderItem.request_info.checkin_date).diff(moment()) > 0;
    const hasPaymentSchedule = Boolean(order.payment_schedule);
    const isPostPaid = Boolean(order.is_post_paid);

    return ReactDOM.createPortal(
        <div className={b()}>
            <div>{getOrderStatusText(status, isTechTexts)}</div>
            <OrderShortInfoIcon
                type="mir"
                active={isMir(order.promo_campaigns?.mir2020)}
            />
            <OrderShortInfoIcon
                type="percentage"
                active={hasPromoCode || hasDiscount}
            />
            <OrderShortInfoIcon type="waiting" active={isFuture} />
            <OrderShortInfoIcon type="calendar" active={hasPaymentSchedule} />
            <OrderShortInfoIcon type="postpay" active={isPostPaid} />
        </div>,
        document.body,
    );
};

export default OrderShortInfo;
