import React, {useState} from 'react';
import {block} from 'bem-cn';
import {compose} from '@bem-react/core';

import MirIcon from 'components/basic/icons/MirIcon';
import InstallmentPlanIcon from 'components/basic/icons/InstallmentPlanIcon';
import WaitingIcon from 'components/basic/icons/WaitingIcon';
import CouponIcon from 'components/basic/icons/CouponIcon';
import PostPayIcon from 'components/basic/icons/PostPayIcon';
import {
    Tooltip as TooltipDesktop,
    withSizeM,
    withViewDefault,
} from '@yandex-lego/components/Tooltip/desktop';

import './index.scss';

const b = block('OrderShortInfo');

const Tooltip = compose(withSizeM, withViewDefault)(TooltipDesktop);

type TOrderShortInfoIcon =
    | 'mir'
    | 'percentage'
    | 'waiting'
    | 'calendar'
    | 'postpay';

interface IOrderShortInfoIconProps {
    active: boolean;
    type: TOrderShortInfoIcon;
}

function getIconByType(type: TOrderShortInfoIcon, active: boolean) {
    switch (type) {
        case 'mir':
            return <MirIcon active={active} />;
        case 'percentage':
            return <CouponIcon active={active} />;
        case 'waiting':
            return <WaitingIcon active={active} />;
        case 'calendar':
            return <InstallmentPlanIcon active={active} />;
        case 'postpay':
            return <PostPayIcon active={active} />;
        default:
            return null;
    }
}

function getIconDescriptionByType(type: TOrderShortInfoIcon) {
    switch (type) {
        case 'mir':
            return 'Заказ участвует в акции МИР';
        case 'percentage':
            return 'Применен промокод';
        case 'waiting':
            return 'Пользователь еще не заселился';
        case 'calendar':
            return 'Заказ в расрочку';
        case 'postpay':
            return 'Заказ с постоплатой';
        default:
            return null;
    }
}

const OrderShortInfoIcon: React.FC<IOrderShortInfoIconProps> = props => {
    const {active, type} = props;

    const iconRef = React.useRef(null);

    const [isVisibleTooltip, setVisibleTooltip] = useState(false);

    return (
        <div className={b('Icon')}>
            <span
                ref={iconRef}
                onMouseOver={() => setVisibleTooltip(true)}
                onMouseLeave={() => setVisibleTooltip(false)}
            >
                {getIconByType(type, active)}
            </span>
            {active && (
                <Tooltip
                    view="default"
                    size="m"
                    anchor={iconRef}
                    visible={isVisibleTooltip}
                >
                    {getIconDescriptionByType(type)}
                </Tooltip>
            )}
        </div>
    );
};

export default OrderShortInfoIcon;
