import React, {useMemo} from 'react';
import moment, {Moment} from 'moment';
import {block} from 'bem-cn';

import {ITrainOrderItem} from 'redux/reducers/types/train';
import {OrderStatus} from 'redux/reducers/types';

import getHumanMoscowFormat from 'lib/time/getHumanMoscowFormat';

import Heading from 'components/basic/Heading/Heading';

const b = block('OrderTrainInfo');

interface IOrderTrainInfoProps {
    className?: string;
    orderStatus: OrderStatus;
    trainOrderItem: ITrainOrderItem;
    hasMaskedInfo: boolean;
    segmentIndex: number | null;
}

const OrderTrainInfo: React.FC<IOrderTrainInfoProps> = props => {
    const {
        className,
        orderStatus,
        trainOrderItem: {
            train_number,
            carrier,
            departure_station,
            arrival_station,
            departure_at,
            arrival_at,
            refund_till,
            eregister_status_till,
            passengers_info,
        },
        segmentIndex,
    } = props;

    const now = useMemo((): Moment => moment(), []);

    const isTrainGone = useMemo((): boolean => {
        return moment(now).isAfter(departure_at);
    }, [departure_at, now]);

    const isTrainArrived = useMemo((): boolean => {
        return moment(now).isAfter(arrival_at);
    }, [arrival_at, now]);

    const isRefundAvailable = useMemo((): boolean => {
        return Boolean(refund_till && moment(now).isAfter(refund_till));
    }, [refund_till, now]);

    const isChangeERAvailable = useMemo((): boolean => {
        return Boolean(
            eregister_status_till && moment(now).isAfter(eregister_status_till),
        );
    }, [eregister_status_till, now]);

    const {totalCost, totalTicketsCost, totalFeeCost, totalInsurance} =
        useMemo(() => {
            return passengers_info.reduce(
                (prevCosts, {ticket, insurance}) => {
                    if (!ticket) {
                        return prevCosts;
                    }

                    const ticketInsuranceAmount =
                        insurance?.partner_operation_status
                            ? insurance.amount.value
                            : 0;

                    return {
                        totalCost:
                            prevCosts.totalCost +
                            ticket.total_cost.value +
                            ticketInsuranceAmount,
                        totalTicketsCost:
                            prevCosts.totalTicketsCost +
                            ticket.tariff_amount.value +
                            ticket.service_amount.value,
                        totalFeeCost:
                            prevCosts.totalFeeCost + ticket.fee_amount.value,
                        totalInsurance:
                            prevCosts.totalInsurance + ticketInsuranceAmount,
                    };
                },
                {
                    totalCost: 0,
                    totalTicketsCost: 0,
                    totalFeeCost: 0,
                    totalInsurance: 0,
                },
            );
        }, [passengers_info]);

    return (
        <div className={b.mix(className)}>
            <Heading level="4">
                Поезд{' '}
                {segmentIndex !== null ? ` (сегмент ${segmentIndex + 1})` : ''}
            </Heading>

            <div>
                🚂 {train_number} · {carrier}
                <br />
                {departure_station} → {arrival_station}
                <br />
                {totalCost ? (
                    <>
                        <span>
                            {`${totalTicketsCost.toFixed(2)} + ${
                                totalInsurance ? `${totalInsurance} + ` : ''
                            }${totalFeeCost.toFixed(2)} = ${totalCost.toFixed(
                                2,
                            )}`}
                        </span>

                        <br />
                    </>
                ) : (
                    ''
                )}
                <span
                    className={`text-${
                        isTrainGone ? 'danger' : 'success'
                    } cursorHelp`}
                    title="Время московское"
                >
                    Отправление: {getHumanMoscowFormat(departure_at)}
                </span>
                <br />
                <span
                    className={`text-${
                        isTrainArrived ? 'danger' : 'success'
                    } cursorHelp`}
                    title="Время московское"
                >
                    Прибытие: {getHumanMoscowFormat(arrival_at)}
                </span>
                {[OrderStatus.OS_FULFILLED, OrderStatus.OS_REFUNDED].includes(
                    orderStatus,
                ) && (
                    <>
                        {refund_till && (
                            <>
                                <br />

                                <span
                                    className={`text-${
                                        isRefundAvailable ? 'danger' : 'success'
                                    } cursorHelp`}
                                    title="Время московское"
                                >
                                    Возврат на сайте до:{' '}
                                    {getHumanMoscowFormat(refund_till)}
                                </span>
                            </>
                        )}

                        {eregister_status_till && (
                            <>
                                <br />

                                <span
                                    className={`text-${
                                        isChangeERAvailable
                                            ? 'danger'
                                            : 'success'
                                    } cursorHelp`}
                                    title="Время московское"
                                >
                                    Изменение ЭР до:{' '}
                                    {getHumanMoscowFormat(
                                        eregister_status_till,
                                    )}
                                </span>
                            </>
                        )}
                    </>
                )}
            </div>
        </div>
    );
};

export default OrderTrainInfo;
