import {useMemo} from 'react';
import {useSelector} from 'react-redux';
import {round} from 'lodash';

import {IStore} from 'redux/reducers/types';

enum EMoneyRefundTypes {
    AMOUNT = 'amount',
}

interface ICalculateRefundWithDiscountParams {
    refundAmount: number;
    discountAmount: number;
    totalAmount: number;
}

interface IUsePartialRefundAmountParams {
    moneyRefundType: EMoneyRefundTypes | string;
    inputValue: number;
}

interface IUsePartialRefundAmountResult {
    partialRefundAmount: number;
    hasDiscount: boolean;
}

const PRECISION = 2;

const calculateRefundWithDiscount = ({
    refundAmount,
    discountAmount,
    totalAmount,
}: ICalculateRefundWithDiscountParams): number =>
    round(refundAmount * (1 - discountAmount / totalAmount), PRECISION);

export const usePartialRefundAmount = ({
    moneyRefundType,
    inputValue,
}: IUsePartialRefundAmountParams): IUsePartialRefundAmountResult => {
    const orderInfo = useSelector((state: IStore) => state.order.item.value);
    const discountAmount =
        orderInfo?.order_price_info.discount_amount.value || 0;
    const originalPrice = orderInfo?.order_price_info.original_price;

    return useMemo(() => {
        if (
            moneyRefundType !== EMoneyRefundTypes.AMOUNT ||
            !discountAmount ||
            !originalPrice
        ) {
            return {partialRefundAmount: inputValue, hasDiscount: false};
        }

        const amount = calculateRefundWithDiscount({
            refundAmount: inputValue,
            discountAmount,
            totalAmount: originalPrice.value,
        });

        return {partialRefundAmount: amount, hasDiscount: true};
    }, [discountAmount, inputValue, moneyRefundType, originalPrice]);
};
