import React, {useEffect, useMemo} from 'react';
import {useDispatch, useSelector} from 'react-redux';
import {useLocation} from 'react-router-dom';
import {Loader} from '@yandex-data-ui/common';
import {block} from 'bem-cn';
import {ordersRequest, resetOrders} from 'redux/reducers/orders/actions';

import {
    isDisplayOrderTypeValid,
    isOrderCarrierValid,
    isOrderPartnerValid,
    isOrderReferralPartnerIdValid,
    isOrderStatusValid,
    isPaymentScheduleTypeValid,
    IStore,
} from 'redux/reducers/types';

import {parseOrdersSearch} from 'lib/url/search';

import Heading from 'components/basic/Heading/Heading';
import OrdersList from 'components/Orders/OrdersList/OrdersList';
import OrdersSearchForm from 'components/Orders/OrdersSearchForm/OrdersSearchForm';

import './index.scss';

const b = block('Orders');

const Orders: React.FC = () => {
    const dispatch = useDispatch();

    const {
        orders: {
            value: orders,
            overallItemsCount,
            isLoading,
            isFetched,
            error,
            isOptionsFetched,
        },
    } = useSelector((state: IStore) => state);

    const location = useLocation();

    const {search} = location;

    const updateOrders = () => {
        const {
            page,
            count,
            partnerId,
            prettyId,
            email,
            phone,
            status,
            displayType,
            fromDate,
            toDate,
            name,
            partner,
            card,
            token,
            sortType,
            isBroken,
            paymentScheduleType,
            ticketNumber,
            yandexUid,
            carrier,
            referralPartnerId,
        } = parseOrdersSearch(search);

        dispatch(
            ordersRequest({
                page: page !== undefined ? Number(page) - 1 : undefined,
                count: count !== undefined ? Number(count) : undefined,
                partnerId,
                prettyId,
                email,
                phone,
                name,
                card,
                token,
                status: isOrderStatusValid(status) ? status : undefined,
                displayType: isDisplayOrderTypeValid(displayType)
                    ? displayType
                    : undefined,
                partner: isOrderPartnerValid(partner) ? partner : undefined,
                fromDate,
                toDate,
                sortType,
                isBroken: isBroken === 'true' ? true : '',
                paymentScheduleType: isPaymentScheduleTypeValid(
                    paymentScheduleType,
                )
                    ? paymentScheduleType
                    : undefined,
                ticketNumber,
                yandexUid,
                carrier: isOrderCarrierValid(carrier) ? carrier : undefined,
                referralPartnerId: isOrderReferralPartnerIdValid(
                    referralPartnerId,
                )
                    ? referralPartnerId
                    : undefined,
            }),
        );
    };

    useEffect(() => {
        if (!search || !isOptionsFetched) {
            return;
        }

        updateOrders();
    }, [search, isOptionsFetched]);

    useEffect(() => {
        if (!search && isFetched) {
            dispatch(resetOrders());
        }
    }, [search, isFetched]);

    useEffect(
        () => () => {
            dispatch(resetOrders());
        },
        [],
    );

    const listWithControls = useMemo(() => {
        if (!isLoading && !isFetched) {
            return null;
        }

        if (isLoading) {
            return (
                <div className={b('loaderWrapper')}>
                    <Loader size="m" />
                </div>
            );
        }

        if (error) {
            return <div className="text-danger">Ошибка поиска</div>;
        }

        if (!orders.length) {
            return <div>Нет подходящих заказов</div>;
        }

        return (
            <div className={b('listWithControls')}>
                <Heading level="3">
                    Найдено заказов: {overallItemsCount}
                </Heading>

                <OrdersList orders={orders} itemsCount={overallItemsCount} />
            </div>
        );
    }, [isLoading, search, location, orders, overallItemsCount]);

    return (
        <div className={b()}>
            <Heading level="1">Поиск заказов</Heading>

            <OrdersSearchForm />

            {listWithControls}
        </div>
    );
};

export default Orders;
