import React, {useMemo} from 'react';
import {block} from 'bem-cn';

import {DEFAULT_ORDER_LOGS_COUNT} from 'constants/search';

import {DisplayOrderType, IBriefOrderInfo} from 'redux/reducers/types';

import getHumanMoscowFormat from 'lib/time/getHumanMoscowFormat';
import getHumanFormatWithoutTime from 'lib/time/getHumanFormatWithoutTime';
import getHumanFormatWithTime from 'lib/time/getHumanFormatWithTime';
import getHumanFormatWithTimeAndTimeZone from 'lib/time/getHumanFormatWithTimeAndTimeZone';
import {getOrderStatusText} from 'lib/dictionaries/orderStatus';
import {getOrderPartnerText} from 'lib/dictionaries/orderPartner';

import Link from 'components/basic/Link/Link';

import './index.scss';

const b = block('OrdersListItem');

const MAP_ORDER_TYPE_TO_HUMAN_TYPE = {
    [DisplayOrderType.DT_UNKNOWN]: 'Неизвестно',
    [DisplayOrderType.DT_AVIA]: 'Авиа',
    [DisplayOrderType.DT_HOTEL]: 'Отели',
    [DisplayOrderType.DT_TRAIN]: 'Поезда',
    [DisplayOrderType.DT_BUS]: 'Автобусы',
    [DisplayOrderType.DT_SUBURBAN]: 'Электрички',
};

interface IOwnProps {
    order: IBriefOrderInfo;
    className?: string;
    cellClassName?: string;
    isTechTexts: boolean;
}

function getPlacesText(placeNumbers?: string[]) {
    if (!placeNumbers) {
        return 'нет';
    }

    if (!placeNumbers.length) {
        return 'без места';
    }

    return placeNumbers.join(', ');
}

const OrdersListItem: React.FC<IOwnProps> = props => {
    const {order, className, cellClassName, isTechTexts} = props;

    const info = useMemo(() => {
        if (order.display_order_type === DisplayOrderType.DT_AVIA) {
            return (
                <div>
                    {order.avia_info.map(({segments}, index) => (
                        <div key={index}>
                            {segments.map(
                                (
                                    {
                                        departure_date_time,
                                        direction,
                                        flight_number,
                                    },
                                    index,
                                ) => (
                                    <div key={index}>
                                        {flight_number} · {direction} ·{' '}
                                        {getHumanFormatWithTime(
                                            departure_date_time,
                                        )}
                                    </div>
                                ),
                            )}
                        </div>
                    ))}
                </div>
            );
        }

        if (order.display_order_type === DisplayOrderType.DT_HOTEL) {
            return (
                <div>
                    {order.hotel_info.map(
                        (
                            {arrive_date, check_out_date, city, hotel_name},
                            index,
                        ) => (
                            <div key={index}>
                                {hotel_name} · {city}
                                <br />
                                {getHumanFormatWithoutTime(arrive_date)} ·{' '}
                                {getHumanFormatWithoutTime(check_out_date)}
                            </div>
                        ),
                    )}
                </div>
            );
        }

        if (order.display_order_type === DisplayOrderType.DT_TRAIN) {
            return (
                <div>
                    {order.train_info.map(
                        (
                            {
                                train_number,
                                car_number,
                                departure_date_time,
                                departure_station,
                                arrival_station,
                            },
                            index,
                        ) => {
                            return (
                                <div key={index}>
                                    🚂 {train_number} 🚃 {car_number}
                                    <br />
                                    {getHumanMoscowFormat(departure_date_time)}
                                    <br />
                                    {departure_station} → {arrival_station}
                                </div>
                            );
                        },
                    )}
                </div>
            );
        }

        if (order.display_order_type === DisplayOrderType.DT_BUS) {
            return (
                <div>
                    {order.bus_info.map(
                        ({route_name, departure, arrival}, index) => {
                            return (
                                <div key={index}>
                                    🚌 {route_name}
                                    <br />
                                    {getHumanFormatWithTimeAndTimeZone(
                                        departure,
                                        'UTC',
                                    )}{' '}
                                    -{' '}
                                    {getHumanFormatWithTimeAndTimeZone(
                                        arrival,
                                        'UTC',
                                    )}
                                </div>
                            );
                        },
                    )}
                </div>
            );
        }

        return <div>нет</div>;
    }, [order.display_order_type, order.avia_info, order.hotel_info]);

    const clients = useMemo(() => {
        if (order.display_order_type === DisplayOrderType.DT_TRAIN) {
            return (
                <div>
                    {order.train_info.map(({passengers}, index) => (
                        <div key={index}>
                            {passengers.map(
                                ({fio, age, place_numbers}, passengerIndex) => (
                                    <div key={passengerIndex}>
                                        {fio}
                                        <br />
                                        👤 {age} 💺{' '}
                                        {getPlacesText(place_numbers)}
                                    </div>
                                ),
                            )}
                        </div>
                    ))}
                </div>
            );
        }

        if (order.display_order_type === DisplayOrderType.DT_BUS) {
            return (
                <div>
                    {order.bus_info.map(({passengers}, index) => (
                        <div key={index}>
                            {passengers.map(({seat_id}, passengerIndex) => (
                                <div key={passengerIndex}>
                                    💺 {seat_id ? seat_id : 'без места'}
                                </div>
                            ))}
                        </div>
                    ))}
                </div>
            );
        }

        return order.clients_names || 'нет';
    }, []);

    return (
        <div className={b.mix(className)}>
            <div className={b('prettyId').mix(cellClassName)}>
                <Link to={`/${order.order_id}`}>{order.pretty_id}</Link>

                {order.broken && <span title="Сломанный заказ"> ⚠️</span>}
            </div>

            <div className={b('date').mix(cellClassName)}>
                {getHumanMoscowFormat(order.created_at)}
            </div>

            <div className={b('type').mix(cellClassName)}>
                {MAP_ORDER_TYPE_TO_HUMAN_TYPE[order.display_order_type]}
            </div>

            <div className={b('partner').mix(cellClassName)}>
                {order.partners
                    .map(partner => getOrderPartnerText(partner, isTechTexts))
                    .join(', ')}
            </div>

            <div
                className={b('status', {tech: isTechTexts}).mix(cellClassName)}
            >
                {isTechTexts && 'Заказа: '}
                {getOrderStatusText(order.display_order_state, isTechTexts)}

                {isTechTexts && (
                    <>
                        <div>Услуги: {order.order_item_state}</div>

                        <div>Платежа: {order.invoice_state || 'нет'}</div>
                    </>
                )}
            </div>

            <div className={b('info').mix(cellClassName)}>{info}</div>

            <div className={b('clients').mix(cellClassName)}>{clients}</div>

            <div className={b('sum').mix(cellClassName)}>
                {order.price_info
                    ? `${order.price_info.value} ${order.price_info.currency}`
                    : 'нет'}
            </div>

            <div className={b('defer').mix(cellClassName)}>
                {order.deferred ? 'да' : 'нет'}
            </div>

            <div className={b('logs').mix(cellClassName)}>
                <Link
                    to={`/${order.order_id}/logs?page=1&count=${DEFAULT_ORDER_LOGS_COUNT}&trimLarge=1`}
                >
                    Логи
                </Link>
            </div>
        </div>
    );
};

export default OrdersListItem;
