import React, {useCallback, useEffect, useState} from 'react';
import {useSelector, useDispatch} from 'react-redux';
import {Select, TextInput, CheckBox} from 'lego-on-react';
import {
    Datepicker,
    DatepickerOutputDates,
    Loader,
} from '@yandex-data-ui/common';
import pick from 'lodash/pick';
import omit from 'lodash/omit';
import {useLocation, useHistory} from 'react-router-dom';
import moment from 'moment-timezone';
import {block} from 'bem-cn';
import {ordersSearchFormOptionsRequest} from 'redux/reducers/orders/actions';

import {DEFAULT_ORDERS_SEARCH_COUNT} from 'constants/search';
import {ORDERS_SEARCH_FORM_FILTER_PARAMS} from 'components/Orders/OrdersSearchForm/helpers/constants';

import {
    DisplayOrderType,
    isOrderCarrierValid,
    isOrderPartnerValid,
    isOrderReferralPartnerIdValid,
    isOrderStatusValid,
    isPaymentScheduleTypeValid,
    IStore,
    OrderCarrier,
    OrderPartner,
    OrderReferralPartnerId,
    OrderStatus,
    PaymentScheduleType,
    PAYMENT_SCHEDULE_TYPES,
    ORDER_CARRIERS,
    ORDER_REFERRAL_PARTNER_IDS,
} from 'redux/reducers/types';

import {extendUrl} from 'lib/url/utils';
import {parseOrdersSearch} from 'lib/url/search';
import getSearchQueryParamsFromState from 'components/Orders/OrdersSearchForm/helpers/getSearchQueryParamsFromState';
import getRecoveredStateFromQuery from 'components/Orders/OrdersSearchForm/helpers/getRecoveredStateFromQuery';
import {getOrderStatusText} from 'lib/dictionaries/orderStatus';
import {getOrderPartnerText} from 'lib/dictionaries/orderPartner';
import {getOrderCarrierText} from 'lib/dictionaries/orderCarrier';
import {getOrderReferralPartnerIdText} from 'lib/dictionaries/orderReferralPartnerId';
import {getPaymentScheduleTypeText} from 'lib/dictionaries/paymentScheduleType';

import ActionButton from 'components/lego/ActionButton/ActionButton';
import OrdersSearchFormOrderType from 'components/Orders/OrdersSearchForm/OrdersSearchFormOrderType/OrdersSearchFormOrderType';

import './index.scss';

export interface IOrdersSearchFormState {
    partnerId: string;
    prettyId: string;
    email: string;
    phone: string;
    status: OrderStatus;
    name: string;
    partner: OrderPartner;
    card: string;
    token: string;
    displayType: DisplayOrderType;
    fromDate: number;
    toDate: number;
    isBroken: boolean;
    paymentScheduleType: PaymentScheduleType;
    ticketNumber: string;
    yandexUid: string;
    carrier: OrderCarrier;
    referralPartnerId: OrderReferralPartnerId;
}

const b = block('OrdersSearchForm');

const OrdersSearchForm: React.FC = () => {
    const location = useLocation();

    const {search} = location;

    const history = useHistory();

    const dispatch = useDispatch();

    const {
        orders: {
            stateOptions,
            displayTypeOptions,
            partnerOptions,
            isOptionsLoading,
            isOptionsFetched,
            optionsError,
        },
        settings: {isTechTexts},
    } = useSelector((store: IStore) => store);

    const [partnerId, setPartnerId] = useState('');
    const [prettyId, setPrettyId] = useState('');
    const [name, setName] = useState('');
    const [email, setEmail] = useState('');
    const [phone, setPhone] = useState('');
    const [partner, setPartner] = useState<OrderPartner>(
        OrderPartner.PT_UNKNOWN,
    );
    const [card, setCard] = useState('');
    const [token, setToken] = useState('');
    const [displayType, setDisplayType] = useState<DisplayOrderType>(
        DisplayOrderType.DT_UNKNOWN,
    );
    const [status, setStatus] = useState<OrderStatus>(OrderStatus.OS_UNKNOWN);
    const [fromDate, setFromDate] = useState(0);
    const [toDate, setToDate] = useState(0);
    const [isBroken, setIsBroken] = useState<boolean>(false);
    const [paymentScheduleType, setPaymentScheduleType] =
        useState<PaymentScheduleType>(PaymentScheduleType.PST_UNKNOWN);
    const [ticketNumber, setTicketNumber] = useState('');
    const [yandexUid, setYandexUid] = useState('');
    const [carrier, setCarrier] = useState<OrderCarrier>(
        OrderCarrier.OC_UNKNOWN,
    );
    const [referralPartnerId, setReferralPartnerId] =
        useState<OrderReferralPartnerId>(OrderReferralPartnerId.ORP_UNKNOWN);

    const recoverOrderState = () => {
        const filterQueryParams = pick(
            parseOrdersSearch(search),
            ORDERS_SEARCH_FORM_FILTER_PARAMS,
        );

        const {
            partnerId,
            prettyId,
            email,
            phone,
            status,
            name,
            partner,
            card,
            token,
            displayType,
            fromDate,
            toDate,
            isBroken,
            paymentScheduleType,
            ticketNumber,
            yandexUid,
            carrier,
            referralPartnerId,
        } = getRecoveredStateFromQuery(filterQueryParams);

        setPartnerId(partnerId);
        setPrettyId(prettyId);
        setEmail(email);
        setPhone(phone);
        setStatus(status);
        setName(name);
        setPartner(partner);
        setCard(card);
        setToken(token);
        setDisplayType(displayType);
        setFromDate(fromDate);
        setToDate(toDate);
        setIsBroken(isBroken);
        setPaymentScheduleType(paymentScheduleType);
        setTicketNumber(ticketNumber);
        setYandexUid(yandexUid);
        setCarrier(carrier);
        setReferralPartnerId(referralPartnerId);
    };

    useEffect(() => {
        if (isOptionsFetched) {
            return;
        }

        dispatch(
            ordersSearchFormOptionsRequest({
                count: 0,
            }),
        );
    }, [isOptionsFetched]);

    useEffect(() => {
        recoverOrderState();
    }, []);

    useEffect(() => {
        if (
            [DisplayOrderType.DT_AVIA, DisplayOrderType.DT_TRAIN].includes(
                displayType,
            )
        ) {
            setPartner(OrderPartner.PT_UNKNOWN);
        }

        if (displayType === DisplayOrderType.DT_AVIA) {
            setPartnerId('');
            setCard('');
            setToken('');
        }
    }, [displayType]);

    const handleSubmit = useCallback(
        (e: React.SyntheticEvent<HTMLFormElement>) => {
            e.preventDefault();

            const prevSearchQueryWithoutFilterParams = omit(
                parseOrdersSearch(search),
                ORDERS_SEARCH_FORM_FILTER_PARAMS,
            );
            const searchFilterQueryParams = getSearchQueryParamsFromState({
                partnerId,
                prettyId,
                email,
                phone,
                status,
                name,
                partner,
                card,
                token,
                displayType,
                fromDate,
                toDate,
                isBroken,
                paymentScheduleType,
                ticketNumber,
                yandexUid,
                carrier,
                referralPartnerId,
            });

            const newSearchUrl = extendUrl(
                {
                    ...location,
                    search: '',
                },
                {
                    ...prevSearchQueryWithoutFilterParams,
                    ...searchFilterQueryParams,
                    page: '1',
                    count: `${DEFAULT_ORDERS_SEARCH_COUNT}`,
                },
            );

            if (newSearchUrl !== `${location.pathname}${location.search}`) {
                history.push(newSearchUrl);
            }
        },
        [
            search,
            partnerId,
            prettyId,
            email,
            displayType,
            fromDate,
            toDate,
            phone,
            status,
            name,
            partner,
            card,
            token,
            isBroken,
            paymentScheduleType,
            ticketNumber,
            yandexUid,
            carrier,
            referralPartnerId,
        ],
    );

    const handleIdChange = useCallback(setPartnerId, []);
    const handlePrettyIdChange = useCallback(setPrettyId, []);
    const handleNameChange = useCallback(setName, []);
    const handleEmailChange = useCallback(setEmail, []);
    const handlePhoneChange = useCallback(setPhone, []);
    const handleCardChange = useCallback(setCard, []);
    const handleTokenChange = useCallback(setToken, []);
    const handleTicketNumberChange = useCallback(setTicketNumber, []);
    const handleYandexUidChange = useCallback(setYandexUid, []);

    const handlePartnerChange = useCallback((newPartner: string | string[]) => {
        setPartner(
            isOrderPartnerValid(newPartner[0])
                ? newPartner[0]
                : OrderPartner.PT_UNKNOWN,
        );
    }, []);

    const handleDisplayTypeChange = useCallback(
        (displayType?: DisplayOrderType): void => {
            setDisplayType(displayType || DisplayOrderType.DT_UNKNOWN);
        },
        [],
    );

    const handleStatusChange = useCallback((newStatus: string | string[]) => {
        setStatus(
            isOrderStatusValid(newStatus[0])
                ? newStatus[0]
                : OrderStatus.OS_UNKNOWN,
        );
    }, []);

    const handleDateRangeChange = useCallback(
        ({from, to}: DatepickerOutputDates) => {
            setFromDate(from ? moment(from).valueOf() : 0);
            setToDate(to ? moment(to).valueOf() : 0);
        },
        [],
    );

    const handleIsBrokenChange = useCallback(() => {
        setIsBroken(!isBroken);
    }, [isBroken]);

    const handlePaymentScheduleTypeChange = useCallback(
        (newSchedule: string | string[]) => {
            setPaymentScheduleType(
                isPaymentScheduleTypeValid(newSchedule[0])
                    ? newSchedule[0]
                    : PaymentScheduleType.PST_UNKNOWN,
            );
        },
        [],
    );

    const handleCarrierChange = useCallback((newCarrier: string | string[]) => {
        setCarrier(
            isOrderCarrierValid(newCarrier[0])
                ? newCarrier[0]
                : OrderCarrier.OC_UNKNOWN,
        );
    }, []);

    const handleReferralPartnerIdChange = useCallback(
        (newReferralPartnerId: string | string[]) => {
            setReferralPartnerId(
                isOrderReferralPartnerIdValid(newReferralPartnerId[0])
                    ? newReferralPartnerId[0]
                    : OrderReferralPartnerId.ORP_UNKNOWN,
            );
        },
        [],
    );

    if (!isOptionsLoading && !isOptionsFetched) {
        return null;
    }

    if (isOptionsLoading) {
        return (
            <div className={b('loaderWrapper')}>
                <Loader size="m" />
            </div>
        );
    }

    if (optionsError) {
        return (
            <div className="text-danger">
                Ошибка загрузки вариантов для поисковой формы
            </div>
        );
    }

    return (
        <div className={b()}>
            <OrdersSearchFormOrderType
                activeType={displayType}
                options={displayTypeOptions}
                onTabClick={handleDisplayTypeChange}
            />

            <form className={b('form')} onSubmit={handleSubmit}>
                <div className={b('formGroup')}>
                    <div className={b('formGroupTitle')}>Дата</div>

                    <Datepicker
                        controlWidth="100%"
                        controlSize="m"
                        from={fromDate}
                        to={toDate}
                        onChange={handleDateRangeChange}
                        timezoneOffset={180}
                        allowNullableValues
                    />
                </div>

                <div className={b('formGroup')}>
                    <div className={b('formGroupTitle')}>Статус</div>

                    <Select
                        theme="normal"
                        size="m"
                        view="default"
                        tone="default"
                        type="radio"
                        width="max"
                        val={status}
                        onChange={handleStatusChange}
                        items={stateOptions.map(option => ({
                            val: option,
                            text: getOrderStatusText(option, isTechTexts),
                        }))}
                    />
                </div>

                <div className={b('formGroup')}>
                    <div className={b('formGroupTitle')}>
                        Номер заказа Путешествий
                    </div>

                    <TextInput
                        theme="normal"
                        size="m"
                        view="default"
                        tone="default"
                        text={prettyId}
                        hasClear
                        onChange={handlePrettyIdChange}
                    />
                </div>

                <div className={b('formGroup')}>
                    <div className={b('formGroupTitle')}>
                        Номер заказа партнёра
                    </div>

                    <TextInput
                        theme="normal"
                        size="m"
                        view="default"
                        tone="default"
                        text={partnerId}
                        disabled={displayType === DisplayOrderType.DT_AVIA}
                        hasClear
                        onChange={handleIdChange}
                    />
                </div>

                <div className={b('formGroup')}>
                    <div className={b('formGroupTitle')}>ФИО</div>

                    <TextInput
                        theme="normal"
                        size="m"
                        view="default"
                        tone="default"
                        text={name}
                        hasClear
                        onChange={handleNameChange}
                    />
                </div>

                <div className={b('formGroup')}>
                    <div className={b('formGroupTitle')}>Почта</div>

                    <TextInput
                        theme="normal"
                        size="m"
                        view="default"
                        tone="default"
                        text={email}
                        hasClear
                        onChange={handleEmailChange}
                    />
                </div>

                <div className={b('formGroup')}>
                    <div className={b('formGroupTitle')}>Телефон</div>

                    <TextInput
                        theme="normal"
                        size="m"
                        view="default"
                        tone="default"
                        text={phone}
                        hasClear
                        onChange={handlePhoneChange}
                    />
                </div>

                <div className={b('formGroup')}>
                    <div className={b('formGroupTitle')}>Партнер</div>

                    <Select
                        theme="normal"
                        size="m"
                        view="default"
                        tone="default"
                        type="radio"
                        width="max"
                        val={partner}
                        onChange={handlePartnerChange}
                        items={partnerOptions.map(option => ({
                            val: option,
                            text: getOrderPartnerText(option, isTechTexts),
                        }))}
                        disabled={[
                            DisplayOrderType.DT_AVIA,
                            DisplayOrderType.DT_TRAIN,
                        ].includes(displayType)}
                    />
                </div>

                <div className={b('formGroup')}>
                    <div className={b('formGroupTitle')}>Yandex UID</div>

                    <TextInput
                        theme="normal"
                        size="m"
                        view="default"
                        tone="default"
                        text={yandexUid}
                        hasClear
                        onChange={handleYandexUidChange}
                    />
                </div>

                <div className={b('formGroup')}>
                    <div className={b('formGroupTitle')}>Маска карты</div>

                    <TextInput
                        theme="normal"
                        size="m"
                        view="default"
                        tone="default"
                        text={card}
                        disabled={displayType === DisplayOrderType.DT_AVIA}
                        hasClear
                        onChange={handleCardChange}
                    />
                </div>

                <div className={b('formGroup')}>
                    <div className={b('formGroupTitle')}>Токен платежа</div>

                    <TextInput
                        theme="normal"
                        size="m"
                        view="default"
                        tone="default"
                        text={token}
                        disabled={displayType === DisplayOrderType.DT_AVIA}
                        hasClear
                        onChange={handleTokenChange}
                    />
                </div>

                <div className={b('formGroup')}>
                    <div className={b('formGroupTitle')}>Тип оплаты</div>

                    <Select
                        theme="normal"
                        size="m"
                        view="default"
                        tone="default"
                        type="radio"
                        width="max"
                        val={paymentScheduleType}
                        onChange={handlePaymentScheduleTypeChange}
                        items={PAYMENT_SCHEDULE_TYPES.map(option => ({
                            val: option,
                            text: getPaymentScheduleTypeText(
                                option,
                                isTechTexts,
                            ),
                        }))}
                        disabled={[DisplayOrderType.DT_SUBURBAN].includes(
                            displayType,
                        )}
                    />
                </div>

                <div className={b('formGroup')}>
                    <div className={b('formGroupTitle')}>Номер билета</div>

                    <TextInput
                        theme="normal"
                        size="m"
                        view="default"
                        tone="default"
                        text={ticketNumber}
                        hasClear
                        onChange={handleTicketNumberChange}
                        disabled={[
                            DisplayOrderType.DT_AVIA,
                            DisplayOrderType.DT_TRAIN,
                            DisplayOrderType.DT_BUS,
                            DisplayOrderType.DT_HOTEL,
                        ].includes(displayType)}
                    />
                </div>

                <div className={b('formGroup')}>
                    <div className={b('formGroupTitle')}>Перевозчик</div>

                    <Select
                        theme="normal"
                        size="m"
                        view="default"
                        tone="default"
                        type="radio"
                        width="max"
                        val={carrier}
                        onChange={handleCarrierChange}
                        items={ORDER_CARRIERS.map(option => ({
                            val: option,
                            text: getOrderCarrierText(option, isTechTexts),
                        }))}
                        disabled={[
                            DisplayOrderType.DT_AVIA,
                            DisplayOrderType.DT_TRAIN,
                            DisplayOrderType.DT_BUS,
                            DisplayOrderType.DT_HOTEL,
                        ].includes(displayType)}
                    />
                </div>

                <div className={b('formGroup')}>
                    <div className={b('formGroupTitle')}>
                        Приводящий партнер
                    </div>

                    <Select
                        theme="normal"
                        size="m"
                        view="default"
                        tone="default"
                        type="radio"
                        width="max"
                        val={referralPartnerId}
                        onChange={handleReferralPartnerIdChange}
                        items={ORDER_REFERRAL_PARTNER_IDS.map(option => ({
                            val: option,
                            text: getOrderReferralPartnerIdText(
                                option,
                                isTechTexts,
                            ),
                        }))}
                        disabled={[
                            DisplayOrderType.DT_AVIA,
                            DisplayOrderType.DT_TRAIN,
                            DisplayOrderType.DT_BUS,
                            DisplayOrderType.DT_SUBURBAN,
                        ].includes(displayType)}
                    />
                </div>

                <div className={b('formGroup', {bottom: true})}>
                    <ActionButton
                        className={b('formSubmit').toString()}
                        type="submit"
                    >
                        Найти
                    </ActionButton>
                </div>
            </form>

            <div className={b('brokenOrdersCheckboxWrapper')}>
                <CheckBox
                    theme="normal"
                    size="m"
                    view="default"
                    tone="default"
                    checked={isBroken}
                    onChange={handleIsBrokenChange}
                >
                    Сломанные заказы
                </CheckBox>
            </div>
        </div>
    );
};

export default OrdersSearchForm;
