import React, {useCallback, useState} from 'react';
import copy from 'copy-to-clipboard';
import {compose} from '@bem-react/core';
import {block} from 'bem-cn';

import {
    Button as LegoButton,
    withViewClear,
} from '@yandex-lego/components/Button/desktop';
import {
    Tooltip as LegoTooltip,
    withSizeM,
    withViewDefault,
} from '@yandex-lego/components/Tooltip/desktop';
import ClipboardButtonIcon from 'components/basic/icons/ClipboardButtonIcon';

import './index.scss';

const b = block('ClipboardButton');
const Button = compose(withViewClear)(LegoButton);
const Tooltip = compose(withSizeM, withViewDefault)(LegoTooltip);

interface IClipboardButtonProps {
    text: string;
    size: 's' | 'm';
}

const ClipboardButton: React.FC<IClipboardButtonProps> = props => {
    const {text, size} = props;
    const [isSuccess, setIsSuccess] = useState(false);
    const [tooltipIsVisible, setTooltipIsVisible] = useState(false);

    const hideTooltip = () => {
        setTooltipIsVisible(false);
    };

    const handleClick = useCallback(() => {
        const result = copy(text);

        setIsSuccess(result);
        setTooltipIsVisible(true);

        setTimeout(hideTooltip, 5000);
    }, [text]);

    const buttonRef = React.useRef(null);
    const state = isSuccess ? 'success' : 'alert';
    const tooltipText = isSuccess
        ? 'Скопировано в буфер'
        : 'Копирование не удалось';

    return (
        <div className={b()}>
            <Button
                view="clear"
                icon={() => <ClipboardButtonIcon size={size} />}
                controlRef={buttonRef}
                onClick={handleClick}
            />
            <Tooltip
                view="default"
                size="m"
                anchor={buttonRef}
                visible={tooltipIsVisible}
                state={state}
                direction={['top-start']}
            >
                {tooltipText}
            </Tooltip>
        </div>
    );
};

export default ClipboardButton;
